<?php

namespace Mnv\Core;

if (!defined('MNV_CMS')) {
    die('Hacking attempt!');
}

use Mnv\Models\Comments;
use Mnv\Models\Feedbacks;
use Mnv\Models\UserGroups;
use Mnv\Core\Auth\Exceptions\UnknownIdException;

/**
 * Class AdminMnv
 * @package Mnv\Core
 */
abstract class AdminMnv extends Mnv
{

    /** @var $managers */
    protected $manager;

    /** @var array */
    private array $menu = [];

    /** @var int  */
    public int $totalPendingComments = 0;

    /** @var array  */
    public array $pendingComments = [];

    /** @var int  */
    public int $totalPendingFeedbacks = 0;

    /** @var array  */
    public array $pendingFeedbacks = [];


    public function __construct()
    {
        global $LANGUAGES;

        parent::__construct();

        $this->initializeMenu();
        $this->initializeAuth($LANGUAGES);
        $this->assignSmartyVariables();

 //        print_r(connect()->queryCount()."\n");
    }

    private function initializeMenu(): void
    {
        try {
            $file = fs_file_get_contents(GLOBAL_ROOT . "/admin/menu/menu.json", "r");
            $this->menu = json_decode($file, true, 512, JSON_THROW_ON_ERROR);
        } catch (\JsonException | \Exception $e) {
            $this->smarty->assign('error',  $e->getMessage());
        }
    }

    private function initializeAuth(array $LANGUAGES): void
    {
        $auth = auth(UserInfo::get_ip(), true, 60 * 5);

        try {
            if ($auth->check() && $auth->doesUserHaveRole($auth->getUserId(), UserGroups::arrayGroup())) {
                $this->manager = $auth->adminManager($auth->getUserId());

                /** получить количество ожидающих комментариев и сами комментарии для   */
                $this->loadPendingComments();
                /** получить количество ожидающих комментариев и сами комментарии для   */
                $this->loadPendingFeedbacks();
                /** определение языка по умолчанию */
                $this->setupLanguages($LANGUAGES);
            } else {
                $auth->logOut();
            }
        } catch (UnknownIdException $e) {
            $auth->logOut();
        }

        if (!$auth->isLoggedIn() || !$auth->hasAnyRole(UserGroups::arrayGroup())) {
            redirect($this->adminUrl . '/auth');
        }
    }

    /** получить количество ожидающих комментариев и сами комментарии для   */
    private function loadPendingComments(): void
    {
        if (ConfigManager::getValue('comments_enabled')) {
            $this->pendingComments = Comments::getPendingComments();
            $this->totalPendingComments = Comments::totalPendingComments();
        }
    }

    /** получить количество ожидающих комментариев и сами комментарии для   */
    private function loadPendingFeedbacks(): void
    {
        $this->pendingFeedbacks = Feedbacks::getPendingFeedbacks();
        $this->totalPendingFeedbacks = Feedbacks::totalPendingFeedbacks();

        $this->smarty->assign('pendingFeedbacks', $this->pendingFeedbacks);
        $this->smarty->assign('totalPendingFeedbacks', $this->totalPendingFeedbacks);

    }

    /** определение языка по умолчанию */
    private function setupLanguages(array $LANGUAGES): void
    {
        foreach ($LANGUAGES as $codename => $tmpLang) {
            $LANGUAGES[$codename] = $tmpLang;
            $LANGUAGES[$codename]['languageTransName'] = lang('languageList:'. $codename);
            if ($tmpLang['isDefault'])
                $this->smarty->assign('DEFAULT_LANG', $codename);
        }
        $this->smarty->assignByRef('LANGUAGES', $LANGUAGES);

    }

    private function assignSmartyVariables(): void
    {
        $this->smarty->assign([
            'module'                => $this->module,
            'menu'                  => $this->getMenu($this->menu),
            'upload_max_filesize'   => $this->upload_max_filesize(),
            'post_max_size'         => $this->post_max_size(),
            'adminUrl'              => $this->adminUrl,
            'adminManager'          => $this->manager,
            'totalPendingComments'  => $this->totalPendingComments,
            'pendingComments'       => $this->pendingComments,
            'statuses'              => lang('general:statuses')
        ]);
    }


    protected function upload_max_filesize(): string
    {
        return formatsize((int) ini_get('upload_max_filesize') * 1024 * 1024);
    }

    protected function post_max_size(): string
    {
        return formatsize((int) ini_get('post_max_size') * 1024 * 1024);
    }


    /**
     * Access rights
     *
     * @param string|null $module
     * @return bool
     */
    protected function permissions(?string $module): bool
    {
        return !empty($module) && !empty($this->manager['permissions']) &&
            in_array($module, $this->manager['permissions'], true) &&
            in_array($this->manager['groupId'], UserGroups::arrayGroup(), true);
    }



    /**
     * Group access rights
     *
     * @return array
     */
    protected function accessRights(): array
    {
        $result = array();
        foreach ($this->menu as $key => $item) {
            $result[$key] = $item;
            $child = array();
            if (isset($item['children'])) {
                foreach ($result[$key]['children'] as $childrenKey => $children) {
                    $child[$childrenKey] = $children;
                }

                unset($result[$key]['children']);

                $result[$key]['children'] = $child;
            }
        }

        return $result;
    }

    /**
     * Формирование меню и получение меню админ панели
     *
     * @param array $menu
     *
     * @return array
     */
    private function getMenu(array $menu): array
    {
        // Если нет разрешений, верните базовое меню
        if (empty($this->manager['permissions'])) {
            foreach ($menu as $key => &$item) {
                if (isset($item['children'])) {
                    // Генерируем `children menu`
                    $item['children'] = $this->generateChildrenMenu($item['children']);
                }
            }
            return $menu;
        }

        foreach ($menu as $key => &$item) {
            if (isset($item['children'])) {
                // Генерируем `children menu`
                $item['children'] = $this->generateChildrenMenu($item['children']);

                // Добавить параметры `developers`, если пользователь является разработчиком
                if ($this->manager['userType'] == UserGroups::DEVELOPER && in_array('managers', $item['activeItems'], true)) {
                    $this->addDeveloperOption($item);
                }
            }
        }

        // Отфильтровать элементы с пустыми children
        $menu = array_filter($menu, function ($item) {
            return !isset($item['children']) || !empty($item['children']);
        });

        // Добавить документацию разработчика
        if ($this->manager['userType'] == UserGroups::DEVELOPER) {
            $menu[] = $this->getDeveloperDocumentation();
        }

        return $menu;
    }

    /**
     * Добавляет запись `developer` в children
     *
     * @param array $item
     *
     * @return void
     */
    private function addDeveloperOption(array &$item): void
    {
        $developer = ['name' => 'Разработчики', 'fileName' => 'developers'];

        // Вставьте `developer` как второго елемента
        array_splice($item['children'], 1, 0, [$developer]);

        // Добавить в activeItems, если его еще нет
        if (!in_array('developers', $item['activeItems'], true)) {
            $item['activeItems'][] = 'developers';
        }
    }

    /**
     * Добавляет пункт меню документации разработчика
     *
     * @return array
     */
    private function getDeveloperDocumentation(): array
    {
        return [
            'activeItems' => ["collection", "database", "docs", "csrf"],
            'name' => 'Документация',
            'icon' => 'ph-question',
            'children' => [
                ['name' => 'Collection', 'fileName' => 'collection'],
                ['name' => 'Database', 'fileName' => 'database'],
                ['name' => 'Smarty', 'fileName' => 'docs'],
                ['name' => 'Security Csrf Token', 'fileName' => 'csrf']
            ]
        ];
    }


    /**
     * Генерирует `children menu` с проверкой доступа
     *
     * @param array $children
     *
     * @return mixed
     */
    private function generateChildrenMenu(array $children): array
    {
        if (empty($this->manager['permissions'])) {
            return $children;
        }

        // Фильтруем дочерние элементы на основе разрешений
        return array_filter($children, function ($child) {
            return $this->access($child['fileName'], $this->manager['permissions']);
        });
    }

    /**
     * Проверяет доступ к контроллеру
     *
     * @param string|null $fileName
     * @param array $permissions
     *
     * @return bool
     */
    private function access(?string $fileName, array $permissions): bool
    {
        return !empty($fileName) && in_array($fileName, $permissions, true);
    }


}