<?php

namespace Mnv\Core\Auth;

use Mnv\Core\Smarty\Design;
use Mnv\Core\Security\Csrf\SecurityCsrfToken;
use Mnv\Core\Security\Csrf\SecurityCsrfSessionProvider;
use Mnv\Core\UserInfo;
use Mnv\Core\ConfigManager;
use Mnv\Models\UserGroups;

/**
 * Class AdminAuthorize
 * @package Mnv\Core\Auth
 */
abstract class AdminAuthorize
{
    /** @var Administration  */
    protected Administration $auth;

    /** @var Design  */
    protected Design $smarty;

    /** @var SecurityCsrfToken */
    protected SecurityCsrfToken $securityCsrfToken;

    /** @var array  */
    protected array $login = [];

    /** @var bool  */
    protected bool $rememberDuration = false;

    /** @var string  */
    protected string $adminUrl = GLOBAL_URL. '/admin';

    /** @var array  */
    protected array $errors = [];

    /** @var array  */
    protected array $messages = [];

    /** @var array  */
    protected array $response = [];

    public function __construct()
    {
        if (!empty(SITE_LANG)) {
            $this->adminUrl = GLOBAL_URL . '/admin/' . SITE_LANG;
        }

        $this->login['login']    = request()->get('login');
        $this->login['password'] = request()->get('password');
        $this->login['remember'] = request()->get('remember');
    }

    /**
     * Initializes authentication and loads language settings.
     */
    protected function initializeAuth(): void
    {
        $this->auth = auth(UserInfo::get_ip(), true, 60 * 5);
        $this->smarty = Design::init()->design();
        loadLanguage(ConfigManager::getValue('admin_language'));

        $this->securityCsrfToken = new SecurityCsrfToken(new SecurityCsrfSessionProvider());

    }

    /** Retrieves the remember duration for the session. */
    protected function getRememberDuration(): void
    {
        $this->rememberDuration = isset($this->login['remember']) && ($this->login['remember'] == 'on' || $this->login['remember'] == 1);
    }

    /**
     * Sanitizes the login data.
     */
    protected function sanitizeLoginData(): void
    {
//        $this->login['phone'] = \Mnv\Core\Utilities\Mask::init()->phone($this->login['phone']);
        $this->login['login'] = htmlspecialchars($this->login['login']);
        $this->login['password'] = htmlspecialchars($this->login['password']);
    }

    /**
     * Handles login error by adding the corresponding error message.
     */
    protected function addLoginError(string $errorKey): void
    {
        $this->errors[] = lang('login:errors:'.$errorKey);
    }

    /**
     * Redirects the admin based on their role.
     */
    protected function redirectToAdminPage(): void
    {
        if ($this->auth->hasAnyRole(array_keys(UserGroups::arrayGroup()))) {
            redirect('/admin/main');
        }
    }


    /**
     * Logs out the admin and redirects to the login page.
     */
    protected function processLogout(): void
    {
        $this->auth->logOut();
        if (!$this->auth->check()) {
            redirect('/admin/auth');
        }
    }


    /**
     * Assigns data to the Smarty template engine.
     */
    protected function assignSmartyVariables(): void
    {
        $this->smarty->assign('login', $this->login);
        if (!empty($this->errors)) {
            $this->smarty->assign('errors', $this->errors);
        }
        if (!empty($this->messages)) {
            $this->smarty->assign('messages', $this->messages);
        }
    }


//    protected function pluralLanguage($value, $type)
//    {
//        // Определение правильной формы в зависимости от числа
//        if ($value % 10 == 1 && $value % 100 != 11) {
//            return lang("login:$type:0"); // Единственное число
//        } elseif ($value % 10 >= 2 && $value % 10 <= 4 && ($value % 100 < 10 || $value % 100 >= 20)) {
//            return lang("login:$type:1");  // Множественное число (2-4)
//        } else {
//            return lang("login:$type:2"); // Множественное число (5+)
//        }
//    }


}