<?php

namespace Mnv\Core\Database;

use Closure;
use Mnv\Core\Collections\Arr;
use Mnv\Core\Test\Log;
use PDO;
use PDOException;


/**
 * Database - Полезный конструктор запросов и класс PDO
 *
 * Class Database
 * @package Mnv\Database
 */
class Connection implements ConnectionInterface
{
    /**
     * Database Version
     *
     * @var string
     */
    const VERSION = '1.6.2';

    /**
     * @var Connection|null
     */
    public $pdo = null;

    /**
     * @var mixed Переменные запроса
     */
    protected $select   = '*';

    protected $from     = null;
    protected $where    = null;
    protected $limit    = null;
    protected $offset   = null;
    protected $join     = null;
    protected $orderBy  = null;
    protected $groupBy  = null;
    protected $having   = null;
    protected $grouped  = false;

    protected $exists   = false;
    protected $selectExists   = '*';

    protected $numRows  = 0;
    protected $insertId = null;
    protected $query    = null;
    protected $error    = null;
    protected $result   = [];
    protected $prefix   = null;
    protected $map      = null;

    protected $indexKey = null;
    protected $valueKey = null;

    /**
     * @var array Операторы SQL
     */
//    protected $operators = ['=', '!=', '<', '>', '<=', '>=', '<>', 'REGEXP', 'AGAINST'];
//    protected $operators = ['=', '!=', '<', '>', '<=', '>=', '<>', 'REGEXP'];
    protected $operators = [
        '=', '<', '>', '<=', '>=', '<>', '!=', '<=>',
        'like', 'like binary', 'not like', 'ilike',
        '&', '|', '^', '<<', '>>',
        'rlike', 'not rlike', 'regexp', 'not regexp',
        '~', '~*', '!~', '!~*', 'similar to',
        'not similar to', 'not ilike', '~~*', '!~~*',
    ];

    /**
     * @var Cache|null
     */
    protected $cache = null;

    /**
     * @var string|null Каталог кеша
     */
    protected $cacheDir = null;

    /**
     * @var int Общее количество запросов
     */
    protected $queryCount = 0;

    /**
     * @var bool
     */
    protected $debug = true;

    /**
     * @var int Общее количество транзакций
     */
    protected $transactionCount = 0;

    private $databaseName;

    private array $_credentials = [
        'dsn'                   => '',
        'driver'                => 'mysql',
        'host'                  => 'localhost',
        'database'              => '',
        'username'              => 'root',
        'password'              => null,
        'charset'               => 'utf8mb4',
        'collation'             => 'utf8mb4_unicode_ci',
        'prefix'                => 'ls_',
        'cacheDir'                => __DIR__ . '/cache/',
        'timestampFormat'       => 'c',
        'readable'              => true,
        'writable'              => true,
        'deletable'             => true,
        'updatable'             => true,
        'return'                => null,
        'debug'                 => false,
        'log'                   => null,
        'profiler'              => false,
    ];

    protected $log;
    /**
     * Database constructor.
     *
     * @param array $config
     */
    public function __construct(array $config)
    {

        $this->log = new Log('sql.log');

        $this->_credentials['driver']       = $config['driver'] ?? 'mysql';
        $this->_credentials['host']         = $config['host'] ?? 'localhost';
        $this->_credentials['username']     = $config['username'] ?? 'root';
        $this->_credentials['password']     = $config['password'] ?? '';
        $this->_credentials['charset']      = $config['charset'] ?? 'utf8mb4';
        $this->_credentials['collation']    = $config['collation'] ?? 'utf8mb4_general_ci';
        $this->_credentials['port']         = $config['port'] ?? (strpos($config['host'], ':') !== false ? explode(':', $config['host'])[1] : '');
        $this->_credentials['prefix']       = $config['prefix'] ?? '';
        $this->_credentials['database']     = $config['database'];
        $this->_credentials['debug']        = $config['debug'] ?? false;
        $this->_credentials['cacheDir']     = $config['cachedir'] ?? __DIR__ . '/cache/';
        $this->databaseName                 = $config['database'];

        $this->debug = $this->_credentials['debug'];

        if (in_array($this->_credentials['driver'], ['', 'mysql', 'pgsql'], true)) {
            $this->_credentials['dsn'] = $this->_credentials['driver'] . ':host=' . str_replace(':' . $this->_credentials['port'], '', $this->_credentials['host']) . ';'
                . ($this->_credentials['port'] !== '' ? 'port=' . $this->_credentials['port'] . ';' : '')
                . 'dbname=' . $this->_credentials['database'];
        }
        elseif ($this->_credentials['driver'] === 'sqlite') {
            $this->_credentials['dsn']  = 'sqlite:' . $this->_credentials['database'];
        }
        elseif ($this->_credentials['driver'] === 'oracle') {
            $this->_credentials['dsn']  = 'oci:dbname=' . $this->_credentials['host'] . '/' . $this->_credentials['database'];
        }

        try {
            $this->pdo = new PDO($this->_credentials['dsn'], $this->_credentials['username'], $this->_credentials['password'], $this->_credentials['options'] ?? null);
            $this->pdo->exec("SET NAMES '" . $this->_credentials['charset'] . "' COLLATE '" . $this->_credentials['collation'] . "'");
            $this->pdo->exec("SET CHARACTER SET '" . $this->_credentials['charset'] . "'");
            $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_OBJ);
        } catch (PDOException $e) {
            die('Cannot the connect to Database with PDO. ' . $e->getMessage());
        }

        return $this->pdo;
    }

    final public function newInstance(array $config = []): Connection
    {
        return new Connection(empty($config) ? $this->_credentials : array_merge($this->_credentials, $config));
    }

    final public function clone(): Connection
    {
        return new self($this->_credentials);
    }
    /**
     * Получить версию MySQL:
     * @return mixed
     */
    public function version()
    {
        return $this->pdo->getAttribute( PDO::ATTR_SERVER_VERSION );
    }

    public function mysqlVersion()
    {
        $res = $this->query("SELECT VERSION() AS `version`")->fetch();
        return $res->version;
    }
   public function mysqlSize()
    {
        $mysql_size = 0;
        $result =  $this->query( "SHOW TABLE STATUS FROM `" . $this->_credentials['database'] . "`" )->fetchAll();
        foreach ($result as $item) {
            if (strpos( $item->Name, $this->_credentials['prefix']) !== false ) {
                $mysql_size += $item->Data_length + $item->Index_length;
            }
        }

        return $mysql_size;
    }

    /**
     * TABLE
     *
     * @param $table
     *
     * @return $this
     */
    public function table($table): Connection
    {
        // Преобразуем строку таблиц в массив, если передана строка
        $tables = is_array($table) ? $table : explode(',', $table);

        // Применяем parseTable для каждой таблицы, минимизируя вызовы trim()
        $this->from = implode(', ', array_map([$this, 'parseTable'], $tables));

//        var_dump($this->from);
        return $this;
    }

    /**
     * SELECT
     *
     * @param array|string $fields
     *
     * @return $this
     */
    public function select($fields): Connection
    {
        $select = is_array($fields) ? implode(', ', $fields) : $fields;
        $this->optimizeSelect($select);

        return $this;
    }

    /**
     * SELECT EXISTS
     *
     * @param array|string $fields
     *
     * @return $this
     */
    public function selectExists($fields): Connection
    {
        $select = is_array($fields) ? implode(', ', $fields) : $fields;
        $this->optimizeSelectExists($select);


        return $this;
    }

    /**
     * MAX
     *
     * @param string      $field
     * @param string|null $name
     *
     * @return $this
     */
    public function max(string $field, ?string $name = null): Connection
    {
        $column = 'MAX(' . $field . ')' . (!is_null($name) ? ' AS ' . $name : '');
        $this->optimizeSelect($column);

        return $this;
    }

    /**
     * MIN
     *
     * @param string      $field
     * @param string|null $name
     *
     * @return $this
     */
    public function min(string $field, ?string $name = null): Connection
    {
        $column = 'MIN(' . $field . ')' . (!is_null($name) ? ' AS ' . $name : '');
        $this->optimizeSelect($column);

        return $this;
    }

    /**
     * SUM
     *
     * @param string      $field
     * @param string|null $name
     *
     * @return $this
     */
    public function sum(string $field, ?string $name = null): Connection
    {
        $column = 'SUM(' . $field . ')' . (!is_null($name) ? ' AS ' . $name : '');
        $this->optimizeSelect($column);

        return $this;
    }

    /**
     * COUNT
     *
     * @param string $field
     * @param string|null $name
     *
     * @return $this
     */
    public function count(string $field = '*', ?string $name = null): Connection
    {
        $column = 'COUNT(' . $field . ')' . (!is_null($name) ? ' AS ' . $name : '');
        $this->optimizeSelect($column);

        return $this;
    }

    /**
     * COUNT DISTINCT
     *
     * @param string $field
     *
     * @return $this
     */
    public function countDistinct(string $field = '*'): Connection
    {
        $column = 'COUNT(DISTINCT ' . $field . ')';
        $this->optimizeSelect($column);

        return $this;
    }

    /**
     * WITH TOTAL COUNT
     *
     * @return $this
     */
    public function withTotalCount(): Connection
    {
        $column = 'SQL_CALC_FOUND_ROWS *';
        $this->optimizeSelect($column);

        return $this;
    }

    /**
     * AVG
     *
     * @param string      $field
     * @param string|null $name
     *
     * @return $this
     */
    public function avg(string $field, ?string $name = null): Connection
    {
        $column = 'AVG(' . $field . ')' . (!is_null($name) ? ' AS ' . $name : '');
        $this->optimizeSelect($column);

        return $this;
    }


    /**
     * JOIN
     *
     * @param string      $table
     * @param string|null $field1
     * @param string|null $operator
     * @param string|null $field2
     * @param string      $type
     *
     * @return $this
     */
    public function join(string $table, string $field1 = null, string $operator = null, string $field2 = null, string $type = ''): Connection
    {
        // Парсим имя таблицы и ее префикс
        $table = $this->parseTable(trim($table));

        // Определяем условие для ON в зависимости от наличия оператора
        if ($operator === null) {
            $on = $field1;
        }
        // Определяем, используется ли допустимый оператор
        else if (in_array($operator, $this->operators, true)) {
            $on = $field1 . ' ' . $operator . ' ' . $field2;
        } else {
            // Если оператор некорректен, используем присвоение
            $on = $field1 . ' = ' . $operator . ($field2 ? ' ' . $field2 : '');
        }

        // Формируем строку JOIN, если это первый JOIN, инициализируем переменную, иначе добавляем к уже существующему JOIN
        if ($this->join === null) {
            $this->join = ' ' . $type . 'JOIN ' . $table . ' ON ' . $on;
        } else {
            $this->join .= ' ' . $type . 'JOIN ' . $table . ' ON ' . $on;
        }

        return $this;

    }

    /**
     * JOIN USING
     *
     * @param string $table
     * @param string|null $field
     * @param string $type
     * @return $this
     */
    public function usingJoin(string $table, string $field = null, string $type = ''): Connection
    {
        // Parse table with alias and clean up spaces
        $table = $this->parseTable($table);

        // If join is null, start building the join clause, otherwise append to the existing one
        $joinClause = ($this->join) ? $this->join : '';

        // Concatenate the JOIN statement efficiently
        $joinClause .= ' ' . $type . 'JOIN ' . $table . ' USING(' . $field . ')';

        // Store the result back in the join property
        $this->join = $joinClause;

        return $this;
    }


    /**
     * INNER JOIN
     *
     * @param string $table
     * @param string $field1
     * @param string $operator
     * @param string $field2
     *
     * @return $this
     */
    public function innerJoin(string $table, string $field1, string $operator = '', string $field2 = ''): Connection
    {
        return $this->join($table, $field1, $operator, $field2, 'INNER ');
    }

    /**
     * LEFT JOIN
     *
     * @param string $table
     * @param string $field1
     * @param string $operator
     * @param string $field2
     *
     * @return $this
     */
    public function leftJoin(string $table, string $field1, string $operator = '', string $field2 = ''): Connection
    {
        return $this->join($table, $field1, $operator, $field2, 'LEFT ');
    }

    /**
     * RIGHT JOIN
     *
     * @param string $table
     * @param string $field1
     * @param string $operator
     * @param string $field2
     *
     * @return $this
     */
    public function rightJoin(string $table, string $field1, string $operator = '', string $field2 = ''): Connection
    {
        return $this->join($table, $field1, $operator, $field2, 'RIGHT ');
    }

    /**
     * FULL OUTER JOIN
     *
     * @param string $table
     * @param string $field1
     * @param string $operator
     * @param string $field2
     *
     * @return $this
     */
    public function fullOuterJoin(string $table, string $field1, string $operator = '', string $field2 = ''): Connection
    {
        return $this->join($table, $field1, $operator, $field2, 'FULL OUTER ');
    }

    /**
     * LEFT OUTER JOIN
     *
     * @param string $table
     * @param string $field1
     * @param string $operator
     * @param string $field2
     *
     * @return $this
     */
    public function leftOuterJoin(string $table, string $field1, string $operator = '', string $field2 = ''): Connection
    {
        return $this->join($table, $field1, $operator, $field2, 'LEFT OUTER ');
    }

    /**
     * RIGHT OUTER JOIN
     *
     * @param string $table
     * @param string $field1
     * @param string $operator
     * @param string $field2
     *
     * @return $this
     */
    public function rightOuterJoin(string $table, string $field1, string $operator = '', string $field2 = ''): Connection
    {
        return $this->join($table, $field1, $operator, $field2, 'RIGHT OUTER ');
    }


    /**
     * GROUPED
     *
     * @param Closure $obj
     *
     * @return $this
     */
    public function grouped(Closure $obj): Connection
    {
        $this->grouped = true;
        call_user_func_array($obj, [$this]);
        $this->where .= ')';

        return $this;
    }

    /**
     * WHERE EXISTS
     *
     * @param  Closure  $obj
     *
     *  SELECT b.bookId, b.sectionId, b.authorId, b.genreId, b.fileId, b.publishedOn, b.title, b.alias, b.url, b.content, b.popularity, b.likes, b.status FROM ls_books AS b
     *  WHERE b.status = 'V' AND
     *  EXISTS (SELECT 1 FROM ls_book_genres AS bg WHERE bg.bookId=b.bookId AND bg.genreId IN (27, 21, 92, 93, 154, 162, 163, 131, 140, 141, 166, 79, 80, 89, 94, 70, 52, 56, 29, 25, 19, 200, 204, 205, 212, 219, 224, 236, 238, 248, 247, 342, 353, 354, 357, 446))
     *  ORDER BY b.popularity DESC LIMIT 10 OFFSET 0
     *
     * @return $this
     */
    public function whereExists(Closure $obj): Connection
    {
        $this->exists = true;
        call_user_func_array($obj, [$this]);
        $this->where .= ')';
        return $this;
    }


    /**
     * WHERE
     *
     * @param array|string $where
     * @param array|string  $operator
     * @param string|null  $val
     * @param string|null $type
     * @param string|null $andOr
     *
     * @return $this
     */
    public function where($where, $operator = null, string $val = null, string $type = '', string $andOr = 'AND'): Connection
    {
        // Проверка на пустые данные сразу
        if (empty($where)) {
            return $this;
        }

        // Обработка массива условий
        if (is_array($where)) {
            $where = $this->processArrayWhere($where, $type, $andOr);
        } else {
            // Обработка условия, если оператор — массив
            $where = $this->processStringWhere($where, $operator, $val, $type);
        }

        // Обработка группировки условий
        if ($this->grouped) {
            $where = '(' . $where;
            $this->grouped = false;
        }

        // Обработка условий для EXISTS
        if ($this->exists) {
            $where = $this->buildExistsCondition($where);
            $this->exists = false;
        }

        // Добавление условия к существующим
        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $andOr . ' ' . $where;

        return $this;
    }

    /**
     * Метод для построения условий из массива
     * @param  array  $where
     * @param  string  $type
     * @param  string  $andOr
     *
     * @return string
     */
    private function processArrayWhere(array $where, string $type, string $andOr): string
    {
        $parts = [];
        foreach ($where as $column => $data) {
            $parts[] = $type . $column . ' = ' . $this->escape($data);
        }
        return implode(" $andOr ", $parts);
    }

    /**
     * Метод для построения условий с подстановкой параметров
     * @param  string  $where
     * @param array|string $operator
     * @param $val
     * @param  string  $type
     *
     * @return string
     */
    private function processStringWhere(string $where, $operator, $val, string $type): string
    {
        // Если оператор является массивом
        if (is_array($operator)) {
            $params = explode('?', $where);
            $whereParts = [];

            foreach ($params as $key => $param) {
                if (!empty($param)) {
                    $whereParts[] = $type . $param . (isset($operator[$key]) ? $this->escape($operator[$key]) : '');
                }
            }
            return implode('', $whereParts);
        }

        // Обрабатываем стандартные операторы и условия
        return (!in_array($operator, $this->operators, true) || $operator === false)
            ? $type . $where . ' = ' . $this->escape($operator)
            : $type . $where . ' ' . $operator . ' ' . $this->escape($val);

    }


    // Метод для построения EXISTS условий
    private function buildExistsCondition(string $where): string
    {
        return ' EXISTS (SELECT ' . $this->selectExists . ' FROM ' . $this->from . ' WHERE ' . preg_replace("/'/", "", $where);
    }


    /**
     * WHERE IN
     *
     * @param string $field
     * @param array  $keys
     * @param string $type
     * @param string $andOr
     *
     * @return $this
     */
    public function whereIn(string $field, array $keys, string $type = '', string $andOr = 'AND'): Connection
    {
        // Проверка на пустой массив
        if (empty($keys)) {
            return $this;
        }

        // Оптимизированное построение массива ключей
        $_keys = array_map(function($v) {
            return is_numeric($v) ? $v : $this->escape($v);
        }, $keys);

        // Построение условия WHERE IN
        $where = $field . ' ' . $type . 'IN (' . implode(', ', $_keys) . ')';

        // Обработка группировки
        if ($this->grouped) {
            $where = '(' . $where;
            $this->grouped = false;
        }

        // Обновление условия WHERE
        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $andOr . ' ' . $where;

        return $this;
    }



    /**
     * OR WHERE
     *
     * @param array|string $where
     * @param string|null  $operator
     * @param string|null  $val
     *
     * @return $this
     */
    public function orWhere($where, string $operator = null, string $val = null): Connection
    {
        return $this->where($where, $operator, $val, '', 'OR');
    }

    /**
     * NOT WHERE
     *
     * @param array|string $where
     * @param string|null  $operator
     * @param string|null  $val
     *
     * @return $this
     */
    public function notWhere($where, string $operator = null, string $val = null): Connection
    {
        return $this->where($where, $operator, $val, 'NOT ', 'AND');
    }

    /**
     * OR NOT WHERE
     *
     * @param array|string $where
     * @param string|null  $operator
     * @param string|null  $val
     *
     * @return $this
     */
    public function orNotWhere($where, string $operator = null, string $val = null): Connection
    {
        return $this->where($where, $operator, $val, 'NOT ', 'OR');
    }

    /**
     * WHERE NULL
     *
     * @param string $where
     * @param bool   $not
     *
     * @return $this
     */
    public function whereNull(string $where, bool $not = false): Connection
    {
        $where .= ' IS ' . ($not ? 'NOT' : '') . ' NULL';
        $this->where = is_null($this->where) ? $where : $this->where . ' ' . 'AND ' . $where;

        return $this;
    }

    /**
     * WHERE IS NULL OR
     *
     * @param string $where
     * @param $operator
     * @param $val
     *
     * @return $this
     */
    public function whereISNullOR(string $where, $operator, $val): Connection
    {
        $where = '(' . $where . ' IS NULL OR ' . $where . ' ' . $operator .' ' . $val . ')';
        $this->where = is_null($this->where) ? $where : $this->where . ' ' . 'AND ' . $where;

        return $this;
    }

    /**
     * WHERE NOT NULL
     *
     * @param string $where
     *
     * @return $this
     */
    public function whereNotNull(string $where): Connection
    {
        return $this->whereNull($where, true);
    }


    /**
     * WHERE AGAINST
     *
     * @param string $where
     * @param string $val
     * @param string $andOr
     *
     * @return $this
     */
    public function whereAgainst(string $where, string $val, string $andOr = 'AND'): Connection
    {
        $where = 'MATCH(' . $where . ') AGAINST (' . $this->escape($val) . ' IN BOOLEAN MODE)' ;
        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $andOr . ' ' . $where;

        return $this;
    }

    /**
     * WHERE ANY
     *
     * SELECT * FROM users WHERE active = true AND (
     *      name LIKE 'Example%' OR
     *      email LIKE 'Example%' OR
     *      phone LIKE 'Example%'
     * )
     *
     * @param  array  $fields
     * @param $operator
     * @param $value
     * @param $boolean
     *
     * @return $this
     */
    public function whereAny(array $fields, $operator = null, $value = null, $boolean = 'and'): Connection
    {
        // Проверка на пустой массив полей
        if (empty($fields)) {
            return $this;
        }

        // Экранируем данные для безопасного использования в запросе
        $escapedData = $this->escape($value);

        // Собираем условия для запроса
        $conditions = [];
        foreach ($fields as $field) {
            $conditions[] = $field . ' ' . $operator . ' ' . $escapedData;
        }

        // Объединяем условия в строку, используя оператор OR
        $where = implode(' OR ', $conditions);

        // Обработка группировки
        if ($this->grouped) {
            $where = '(' . $where; // Сбрасываем флаг группировки
            $this->grouped = false;
        } else {
            $where = '(' . $where . ')';
        }

        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $boolean . ' ' . $where;

        return $this;
    }

    /**
     * WHERE NOT
     *
     * SELECT * FROM albums WHERE published = true AND NOT (
     *      title LIKE '%explicit%' OR
     *      lyrics LIKE '%explicit%' OR
     *      tags LIKE '%explicit%'
     * )
     *
     * @param  array  $fields
     * @param $operator
     * @param $value
     * @param $boolean
     *
     * @return $this
     */
    public function whereNone(array $fields, $operator = null, $value = null, $boolean = 'and'): Connection
    {
        // Проверка на пустой массив полей
        if (empty($fields)) {
            return $this;
        }

        // Экранируем данные для безопасного использования в запросе
        $escapedData = $this->escape($value);

        // Собираем условия для запроса
        $conditions = [];
        foreach ($fields as $field) {
            $conditions[] = $field . ' ' . $operator . ' ' . $escapedData;
        }

        // Объединяем условия в строку, используя оператор OR
        $where = implode(' OR ', $conditions);

        // Обработка группировки
        if ($this->grouped) {
            $where = '(' . $where; // Сбрасываем флаг группировки
            $this->grouped = false;
        } else {
            $where = '(' . $where . ')';
        }

        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $boolean . ' ' . $where;

        return $this;
    }


    /**
     * IN
     *
     * @param string $field
     * @param array  $keys
     * @param string $type
     * @param string $andOr
     *
     * @return $this
     */
    public function in(string $field, array $keys, string $type = '', string $andOr = 'AND'): Connection
    {
        return $this->whereIn($field, $keys, $type, $andOr);
    }

    /**
     * NOT IN
     *
     * @param string $field
     * @param array  $keys
     *
     * @return $this
     */
    public function notIn(string $field, array $keys): Connection
    {
        return $this->whereIn($field, $keys, 'NOT ', 'AND');
    }

    /**
     * OR IN
     *
     * @param string $field
     * @param array  $keys
     *
     * @return $this
     */
    public function orIn(string $field, array $keys): Connection
    {
        return $this->whereIn($field, $keys, '', 'OR');
    }

    /**
     * OR NOT IN
     *
     * @param string $field
     * @param array  $keys
     *
     * @return $this
     */
    public function orNotIn(string $field, array $keys): Connection
    {
        return $this->whereIn($field, $keys, 'NOT ', 'OR');
    }


    /**
     * FIND IN SET
     *
     * @param string         $field
     * @param string|integer $key
     * @param string|null    $type
     * @param string         $andOr
     *
     * @return $this
     */
    public function findInSet(string $field, $key, ?string $type = '', string $andOr = 'AND'): Connection
    {
        $key = is_numeric($key) ? $key : $this->escape($key);
        $where =  $type . 'FIND_IN_SET (' . $key . ', '.$field.')';

        if ($this->grouped) {
            $where = '(' . $where;
            $this->grouped = false;
        }

        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $andOr . ' ' . $where;

        return $this;
    }

    /**
     * NOT FIND IN SET
     *
     * @param string $field
     * @param string $key
     *
     * @return $this
     */
    public function notFindInSet(string $field, string $key): Connection
    {
        return $this->findInSet($field, $key, 'NOT ');
    }

    /**
     * OR FIND IN SET
     *
     * @param string $field
     * @param string $key
     *
     * @return $this
     */
    public function orFindInSet(string $field, string $key): Connection
    {
        return $this->findInSet($field, $key, '', 'OR');
    }

    /**
     * OR NOT FIND IN SET
     *
     * @param string $field
     * @param string $key
     *
     * @return $this
     */
    public function orNotFindInSet(string $field, string $key): Connection
    {
        return $this->findInSet($field, $key, 'NOT ', 'OR');
    }

    /**
     * BETWEEN
     *
     * @param string     $field
     * @param string|int $value1
     * @param string|int $value2
     * @param string     $type
     * @param string     $andOr
     *
     * @return $this
     */
    public function between(string $field, $value1, $value2, string $type = '', string $andOr = 'AND'): Connection
    {
        $where = '(' . $field . ' ' . $type . 'BETWEEN ' . ($this->escape($value1) . ' AND ' . $this->escape($value2)) . ')';

        if ($this->grouped) {
            $where = '(' . $where;
            $this->grouped = false;
        }

        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $andOr . ' ' . $where;

        return $this;
    }

    /**
     * NOT BETWEEN
     *
     * @param string     $field
     * @param string|int $value1
     * @param string|int $value2
     *
     * @return $this
     */
    public function notBetween(string $field, $value1, $value2): Connection
    {
        return $this->between($field, $value1, $value2, 'NOT ', 'AND');
    }

    /**
     * OR BETWEEN
     *
     * @param string     $field
     * @param string|int $value1
     * @param string|int $value2
     *
     * @return $this
     */
    public function orBetween(string $field, $value1, $value2): Connection
    {
        return $this->between($field, $value1, $value2, '', 'OR');
    }

    /**
     * OR NOT BETWEEN
     *
     * @param string     $field
     * @param string|int $value1
     * @param string|int $value2
     *
     * @return $this
     */
    public function orNotBetween(string $field, $value1, $value2): Connection
    {
        return $this->between($field, $value1, $value2, 'NOT ', 'OR');
    }

    /**
     * LIKE
     *
     * @param string $field
     * @param string $data
     * @param string $type
     * @param string $andOr
     *
     * @return $this
     */
    public function like(string $field, string $data, string $type = '', string $andOr = 'AND'): Connection
    {
        $like = $this->escape($data);
        $where = $field . ' ' . $type . 'LIKE ' . $like;

        if ($this->grouped) {
            $where = '(' . $where;
            $this->grouped = false;
        }

        $this->where = is_null($this->where) ? $where : $this->where . ' ' . $andOr . ' ' . $where;

        return $this;
    }


    /**
     * OR LIKE
     *
     * @param string $field
     * @param string $data
     *
     * @return $this
     */
    public function orLike(string $field, string $data): Connection
    {
        return $this->like($field, $data, '', 'OR');
    }

    /**
     * NOT LIKE
     *
     * @param string $field
     * @param string $data
     *
     * @return $this
     */
    public function notLike(string $field, string $data): Connection
    {
        return $this->like($field, $data, 'NOT ', 'AND');
    }

    /**
     * OR NOT LIKE
     *
     * @param string $field
     * @param string $data
     *
     * @return $this
     */
    public function orNotLike(string $field, string $data): Connection
    {
        return $this->like($field, $data, 'NOT ', 'OR');
    }

    /**
     * LIMIT
     *
     * @param int      $limit
     * @param int|null $limitEnd
     *
     * @return $this
     */
    public function limit(int $limit, int $limitEnd = null): Connection
    {
        $this->limit = !is_null($limitEnd) ? $limit . ', ' . $limitEnd : $limit;

        return $this;
    }

    /**
     * OFFSET
     *
     * @param int $offset
     *
     * @return $this
     */
    public function offset(int $offset): Connection
    {
        $this->offset = $offset;

        return $this;
    }

    /**
     * PAGINATION
     *
     * @param int $perPage
     * @param int $page
     *
     * @return $this
     */
    public function pagination(int $perPage, int $page): Connection
    {
        $this->limit = $perPage;
        $this->offset = (($page > 0 ? $page : 1) - 1) * $perPage;

        return $this;
    }

    /**
     * ORDER BY
     * @param string      $orderBy
     * @param string|null $orderDir
     *
     * @return $this
     */
    public function orderBy(string $orderBy, string $orderDir = null): Connection
    {
        if (!is_null($orderDir)) {
            $this->orderBy = $orderBy . ' ' . strtoupper($orderDir);
        } else {
            $this->orderBy = stristr($orderBy, ' ') || strtolower($orderBy) === 'rand()' ? $orderBy : $orderBy . ' ASC';
        }

        return $this;
    }

    /**
     * GROUP BY
     *
     * @param string|array $groupBy
     *
     * @return $this
     */
    public function groupBy($groupBy): Connection
    {
        $this->groupBy = is_array($groupBy) ? implode(', ', $groupBy) : $groupBy;

        return $this;
    }

    /**
     * HAVING
     *
     * @param string            $field
     * @param string|array|null $operator
     * @param string|null       $val
     *
     * @return $this
     */
    public function having(string $field, $operator = null, string $val = null): Connection
    {
        if (is_array($operator)) {
            $fields = explode('?', $field);
            $where = '';
            foreach ($fields as $key => $value) {
                if (!empty($value)) {
                    $where .= $value . (isset($operator[$key]) ? $this->escape($operator[$key]) : '');
                }
            }
            $this->having = $where;
        } elseif (!in_array($operator, $this->operators, true)) {
            $this->having = $field . ' > ' . $this->escape($operator);
        } else {
            $this->having = $field . ' ' . $operator . ' ' . $this->escape($val);
        }

        return $this;
    }

    /**
     * NUM ROWS
     *
     * @return int
     */
    public function numRows(): int
    {
        return $this->numRows;
    }

    /**
     * INSERT ID
     * @return int|null
     */
    public function insertId(): ?int
    {
        return $this->insertId;
    }

    /**
     * ERROR
     *
     * @throw PDOException
     */
    public function error(): void
    {
        if ($this->debug === true) {
            if (php_sapi_name() === 'cli') {
                die("Query: " . $this->query . PHP_EOL . "Error: " . $this->error . PHP_EOL);
            }

            $msg = '<h1>Database Error</h1>';
            $msg .= '<h4>Query: <em style="font-weight:normal;">"' . $this->query . '"</em></h4>';
            $msg .= '<h4>Error: <em style="font-weight:normal;">' . $this->error . '</em></h4>';

            die($msg);
        }

        throw new PDOException($this->error . '. (' . $this->query . ')');
    }

    /**
     * Получает первое значение из результата запроса
     *
     * @param string|null $argument
     *
     * @return mixed
     */
    public function getValue($argument = null)
    {
        // Ограничиваем количество возвращаемых строк до 1
        $this->limit = 1;

        // Выполняем запрос
        $query = $this->getAll(true);
        $result = $this->query($query, true, 'array', $argument);

        // Проверяем результат и возвращаем первое значение
        return $this->extractFirstValue($result);

    }

    /**
     * Извлекает первое значение из результата
     *
     * @param array|null $result
     * @return mixed|null
     */
    private function extractFirstValue(array $result)
    {
        // Если результат существует и содержит хотя бы одну строку
        if (!empty($result) && isset($result[0])) {
            $firstRow = $result[0];

            // Возвращаем первое значение
            if (is_array($firstRow)) {
                return reset($firstRow); // Быстрое получение первого значения из массива
            }

            // Для объектов возвращаем первое публичное свойство
            $array = (array) $firstRow;
            return $firstRow ? reset($array) : null;
        }

        return null;
    }


    /**
     * @param string|bool $type
     * @param string|null $argument
     *
     * @return mixed
     */
    public function get($type = null, $argument = null)
    {
        $this->limit = 1;
        $query = $this->getAll(true);

        return $type === true ? $query : $this->query($query, false, $type, $argument);
    }

    /**
     * @param bool|string $type
     * @param string|null $argument
     *
     * @return mixed
     */
    public function getAll($type = null, $argument = null)
    {
        $query = 'SELECT ' . $this->select . ' FROM ' . $this->from;

        if (!is_null($this->join)) {
            $query .= $this->join;
        }

        if (!is_null($this->where)) {
            $query .= ' WHERE ' . $this->where;
        }

        if (!is_null($this->groupBy)) {
            $query .= ' GROUP BY ' . $this->groupBy;
        }

        if (!is_null($this->having)) {
            $query .= ' HAVING ' . $this->having;
        }

        if (!is_null($this->orderBy)) {
            $query .= ' ORDER BY ' . $this->orderBy;
        }

        if (!is_null($this->limit)) {
            $query .= ' LIMIT ' . $this->limit;
        }

        if (!is_null($this->offset)) {
            $query .= ' OFFSET ' . $this->offset;
        }

        return $type === true ? $query : $this->query($query, true, $type, $argument);
    }

    /**
     * @param array $data
     * @param bool  $type
     *
     * @return bool|string|int|null
     */
    public function insert(array $data, bool $type = false)
    {
        $query = 'INSERT INTO ' . $this->from;

        $values = array_values($data);
        if (isset($values[0]) && is_array($values[0])) {
            $column = implode(', ', array_keys($values[0]));
            $query .= ' (' . $column . ') VALUES ';
            foreach ($values as $value) {
                $val = implode(', ', array_map([$this, 'escape'], $value));
                $query .= '(' . $val . '), ';
            }
            $query = trim($query, ', ');
        } else {
            $column = implode(', ', array_keys($data));
            $val = implode(', ', array_map([$this, 'escape'], $data));
            $query .= ' (' . $column . ') VALUES (' . $val . ')';
        }

        if ($type === true) {
            return $query;
        }

        if ($this->query($query, false)) {
            $this->insertId = $this->pdo->lastInsertId();
            return $this->insertId();
        }

        return false;
    }

    /**
     * REPLACE INTO
     * @param array $data
     * @param bool  $type
     *
     * @return bool|string|int|null
     */
    public function replace(array $data, bool $type = false)
    {
        $query = 'REPLACE INTO ' . $this->from;

        $values = array_values($data);
        if (isset($values[0]) && is_array($values[0])) {
            $column = implode(', ', array_keys($values[0]));
            $query .= ' (' . $column . ') VALUES ';
            foreach ($values as $value) {
                $val = implode(', ', array_map([$this, 'escape'], $value));
                $query .= '(' . $val . '), ';
            }
            $query = trim($query, ', ');
        } else {
            $column = implode(', ', array_keys($data));
            $val = implode(', ', array_map([$this, 'escape'], $data));
            $query .= ' (' . $column . ') VALUES (' . $val . ')';
        }

        if ($type === true) {
            return $query;
        }

        if ($this->query($query, false)) {
            $this->insertId = $this->pdo->lastInsertId();
            return $this->insertId();
        }

        return false;
    }

    /**
     *
     * Вставка или обновление данных с помощью UPSERT
     *
     * INSERT INTO ON DUPLICATE KEY UPDATE
     * использвать без AUTO_INCREMENT
     *
     * EXAMPLE
     *
     * INSERT INTO ls_users_throttling (bucket, tokens, replenished_at, expires_at) VALUES ('ehZWvLHPFmkZjwlMOl-s5dN9Xwn7rCX33vJlmdmlsHI', 74.03, 1727443657, 1727983657)
     * ON DUPLICATE KEY UPDATE tokens = 74.03, replenished_at = 1727443657, expires_at = 1727983657;
     *
     * INSERT INTO ls_user_login_attempts (deviceId, login, visitorIp, attempts, last_attempt) VALUES ('ed31cbb5-a220-7b15-b8d0-2e02496173cd', 'ivanov', '::1', 1, '2024-09-27 19:42:37')
     * ON DUPLICATE KEY UPDATE attempts = attempts + 1, login = 'ivanov', last_attempt = '2024-09-27 19:42:37'
     *
     * @param array $data
     * @param bool  $type
     *
     * @return bool|string|int|null
     *

     *
     */
    public function upsert(array $data, array $updateFields = [], bool $type = false)
    {
        // Prepare the INSERT part of the query
        $columns = implode(', ', array_keys($data));
        $values = implode(', ', array_map([$this, 'escape'], array_values($data)));

        // Build the base INSERT query
        $query = 'INSERT INTO ' . $this->from . ' (' . $columns . ') VALUES (' . $values . ') ';

        // Prepare the ON DUPLICATE KEY UPDATE part
        $update = [];
        if (!empty($updateFields)) {
            foreach ($updateFields as $key => $value) {
                // for increment - пример (attempts + 1)
                if (is_string($value) && preg_match('/[+]\s*\d+/', trim($value))) {
                    $update[] = $key . ' = '. $value;
                } else {
                    $update[] = $key . ' = '. $this->escape($value);
                }
            }
        } else {
            foreach ($data as $key => $value) {
                $update[] = "$key = VALUES($key)";
            }
        }

        if (!empty($update)) {
            $query .= 'ON DUPLICATE KEY UPDATE ' . implode(', ', $update);
        }

        if ($type === true) {
            return $query;
        }

        // Execute the query
        if ($this->query($query, false)) {
            $this->insertId = $this->pdo->lastInsertId();
            return $this->insertId;
        }

        return false;
    }


    /**
     * @param array $data
     * @param bool  $type
     *
     * @return mixed|string
     */
    public function update(array $data, bool $type = false)
    {
        $query = 'UPDATE ' . $this->from . ' SET ';

        $values = [];
        foreach ($data as $column => $val) {
            $values[] = $column . '=' . $this->escape($val);
        }
        $query .= implode(',', $values);



        if (!is_null($this->where)) {
            $query .= ' WHERE ' . $this->where;
        }

        if (!is_null($this->orderBy)) {
            $query .= ' ORDER BY ' . $this->orderBy;
        }

        if (!is_null($this->limit)) {
            $query .= ' LIMIT ' . $this->limit;
        }

        return $type === true ? $query : $this->query($query, false);
    }

    /**
     * @param array $data
     * @param bool  $type
     *
     * @return mixed|string
     */
    public function updateLow(array $data, bool $type = false)
    {
        $query = 'UPDATE LOW_PRIORITY ' . $this->from . ' SET ';
        $values = [];

        foreach ($data as $column => $val) {
            $values[] = $column . '=' . $this->escape($val);
        }
        $query .= implode(',', $values);

        if (!is_null($this->where)) {
            $query .= ' WHERE ' . $this->where;
        }

        if (!is_null($this->orderBy)) {
            $query .= ' ORDER BY ' . $this->orderBy;
        }

        if (!is_null($this->limit)) {
            $query .= ' LIMIT ' . $this->limit;
        }

        return $type === true ? $query : $this->query($query, false);
    }

    /**
     * @return mixed
     * UPDATE `table` SET `url` = CONCAT('/about2', '/', fileName, '.htm') WHERE `sectionId` = '2'
     */
    public function updateConcat($field, $concat)
    {
        $query = 'UPDATE ' . $this->from . ' SET ' . $field . '=' . $concat;

        if (!is_null($this->where)) {
            $query .= ' WHERE ' . $this->where;
        }

        if (!is_null($this->orderBy)) {
            $query .= ' ORDER BY ' . $this->orderBy;
        }

        if (!is_null($this->limit)) {
            $query .= ' LIMIT ' . $this->limit;
        }

        return $this->query($query, false);
    }


    /**
     * @param string $field
     * @param int $amount
     * @param bool  $type
     *
     * @return mixed|string
     */
    public function increment(string $field, int $amount = 1 , bool $type = false)
    {
        $query = 'UPDATE ' . $this->from . ' SET ';

        $values[] = $field . '=' . $field  . ' + ' . $this->escape($amount);

        $query .= implode(',', $values);

        if (!is_null($this->where)) {
            $query .= ' WHERE ' . $this->where;
        }

        if (!is_null($this->orderBy)) {
            $query .= ' ORDER BY ' . $this->orderBy;
        }

        if (!is_null($this->limit)) {
            $query .= ' LIMIT ' . $this->limit;
        }

        return $type === true ? $query : $this->query($query, false);
    }

    public function decrement(string $field, int $amount = 1 , bool $type = false)
    {
        $query = 'UPDATE ' . $this->from . ' SET ';

        $values[] = $field . '=' . $field  . ' - ' . $this->escape($amount);

        $query .= implode(',', $values);

        if (!is_null($this->where)) {
            $query .= ' WHERE ' . $this->where;
        }

        if (!is_null($this->orderBy)) {
            $query .= ' ORDER BY ' . $this->orderBy;
        }

        if (!is_null($this->limit)) {
            $query .= ' LIMIT ' . $this->limit;
        }

        return $type === true ? $query : $this->query($query, false);
    }

    /**
     * @param bool $type
     *
     * @return mixed|string
     */
    public function delete(bool $type = false)
    {
        $query = 'DELETE FROM ' . $this->from;

        if (!is_null($this->where)) {
            $query .= ' WHERE ' . $this->where;
        }

        if (!is_null($this->orderBy)) {
            $query .= ' ORDER BY ' . $this->orderBy;
        }

        if (!is_null($this->limit)) {
            $query .= ' LIMIT ' . $this->limit;
        }

        if ($query === 'DELETE FROM ' . $this->from) {
            $query = 'TRUNCATE TABLE ' . $this->from;
        }

        return $type === true ? $query : $this->query($query, false);
    }

    /**
     * @return mixed
     */
    public function analyze()
    {
        return $this->query('ANALYZE TABLE ' . $this->from, false);
    }

    /**
     * @return mixed
     */
    public function check()
    {
        return $this->query('CHECK TABLE ' . $this->from, false);
    }

    /**
     * @return mixed
     */
    public function checksum()
    {
        return $this->query('CHECKSUM TABLE ' . $this->from, false);
    }

    /**
     * @return mixed
     */
    public function optimize()
    {
        return $this->query('OPTIMIZE TABLE ' . $this->from, false);
    }

    /**
     * @return mixed
     */
    public function repair()
    {
        return $this->query('REPAIR TABLE ' . $this->from, false);
    }


    /**
     * @return bool
     */
    public function transaction(): bool
    {
        if (!$this->transactionCount++) {
            return $this->pdo->beginTransaction();
        }

        $this->pdo->exec('SAVEPOINT trans' . $this->transactionCount);
        return $this->transactionCount >= 0;
    }

    /**
     * фиксирует текущую транзакцию, делая ее изменения постоянными.
     * @return bool
     */
    public function commit(): bool
    {
        if (!--$this->transactionCount) {
            return $this->pdo->commit();
        }

        return $this->transactionCount >= 0;
    }

    /**
     * откатывает текущую транзакцию, отменяя ее изменения.
     * @return bool
     */
    public function rollBack(): bool
    {
        if (--$this->transactionCount) {
            $this->pdo->exec('ROLLBACK TO trans' . ($this->transactionCount + 1));
            return true;
        }

        return $this->pdo->rollBack();
    }

    /**
     * @return mixed
     */
    public function exec()
    {
        if (is_null($this->query)) {
            return null;
        }

        $query = $this->pdo->exec($this->query);
        if ($query === false) {
            $this->error = $this->pdo->errorInfo()[2];
            $this->error();
        }

        return $query;
    }



    /**
     * @param string|null $type
     * @param string|null $argument
     * @param bool   $params
     *
     * @return mixed
     */
    public function fetch(string $type = null, string $argument = null, bool $params = false)
    {
        if (is_null($this->query)) {
            return null;
        }

        $query = $this->pdo->query($this->query);
        if (!$query) {
            $this->error = $this->pdo->errorInfo()[2];
            $this->error();
        }

        $type = $this->getFetchType($type);
        if ($type === PDO::FETCH_CLASS) {
            $query->setFetchMode($type, $argument);
        } else {
            $query->setFetchMode($type);
        }

        $result = $params ? $query->fetchAll() : $query->fetch();
        $this->numRows = is_array($result) ? count($result) : 1;

        return $result;
    }

    /**
     * @param string|null $type
     * @param string|null $argument
     *
     * @return mixed
     */
    public function fetchAll(string $type = null, string $argument = null)
    {
        return $this->fetch($type, $argument, true);
    }

    /**
     * @param string|bool $type
     * @param string|null $argument
     *
     * @return mixed
     */
//    public function getAllIndexes($type = null, $argument = null)
//    {
//
//        $query = $this->getAll(true);
//
//        $result = $this->query($query, true, $type, $argument);
//
//        if (is_null($this->indexKey)) {
//            if (is_null($this->valueKey)) {
//                $this->result = $result;
//            } else {
//                foreach($result as $row) {
//                    $rows[] = is_array($row) ? $row[$this->valueKey] : $row->{$this->valueKey};
//                }
//            }
//        } elseif (is_null($this->valueKey)) {
//            foreach ($result as $row) {
//                if (is_array($row)) {
//                    $rows[$row[$this->indexKey]] = $row;
//                } else {
//                    $rows[$row->{$this->indexKey}] = $row;
//                }
//            }
//        } else {
//            foreach($result as $row) {
//                if (is_array($row)) {
//                    $rows[$row[$this->indexKey]] = $row[$this->valueKey];
//                } else {
//                    $rows[$row->{$this->indexKey}] = $row->{$this->valueKey};
//                }
//            }
//        }
//
//        if (!empty($rows)) {
//            $this->resetIndexes();
//            return $this->result = $rows;
//        }
//
//        return null;
//    }

//    public function getAllIndexes($type = null, $argument = null)
//    {
//        // Получаем все данные
//        $query = $this->getAll(true);
//        $result = $this->query($query, true, $type, $argument);
//
//        // Обрабатываем результат в зависимости от наличия indexKey и valueKey
//        if (is_null($this->indexKey)) {
//            $this->result = is_null($this->valueKey) ? $result : $this->extractValues($result, $this->valueKey);
//        } else {
//            $this->result = $this->mapResultToIndex($result, $this->indexKey, $this->valueKey);
//        }
//
//        // Если результат не пуст, сбрасываем индексы и возвращаем результат
//        if (!empty($this->result)) {
//            $this->resetIndexes();
//            return $this->result;
//        }
//
//        return null;
//    }
//
//// Извлечение значений из результата по ключу
//    private function extractValues($result, $key)
//    {
//        $rows = [];
//        foreach ($result as $row) {
//            $rows[] = is_array($row) ? $row[$key] : $row->{$key};
//        }
//        return $rows;
//    }
//
//// Привязка результатов к индексам
//    private function mapResultToIndex($result, $indexKey, $valueKey = null)
//    {
//        $rows = [];
//        if (!empty($result)) {
//            foreach ($result as $row) {
//                $index = is_array($row) ? $row[$indexKey] : $row->{$indexKey};
//                $value = is_null($valueKey) ? $row
//                    : (is_array($row) ? $row[$valueKey] : $row->{$valueKey});
//                $rows[$index] = $value;
//            }
//        }
//        return $rows;
//    }


    /**
     * @param null $indexKey
     * @return $this
     */
    public function indexKey($indexKey = null): Connection
    {
        $this->indexKey = $indexKey;
        return $this;
    }

    /**
     * @param null $valueKey
     * @return $this
     */
    public function valueKey($valueKey = null): Connection
    {
        $this->valueKey = $valueKey;
        return $this;
    }

    /**
     * PLUCK
     *
     * @param string $value
     * @param $key
     *
     * @return array
     */
    public function pluck(string $value, $key = null): array
    {
        $query = $this->getAll(true);
        $items = $this->query($query, true, 'array', null);

        return !empty($items) ? Arr::pluck($items, $value, $key) : [];
    }

    /**
     * FIRST
     *
     * @param  callable|null  $callback
     * @param $default
     *
     * @return mixed
     */
    public function first(callable $callback = null, $default = null)
    {
        $query = $this->getAll(true);
        $items = $this->query($query, true, 'array', null);

        return Arr::first($items, $callback, $default);
    }

    /**
     * MAP
     *
     * @param  callable  $callback
     *
     * @return array
     */
    public function map(callable $callback): array
    {
        $query = $this->getAll(true);
        $items = $this->query($query, true, null, null);

        return Arr::map($items, function ($item) use ($callback) {
            return $callback(...$item);
        });

    }

    /**
     * KEY BY
     * @param $keyBy
     *
     * @return array|null
     */
    public function keyBy($keyBy): ?array
    {
        $query = $this->getAll(true);
        $items = $this->query($query, true, 'array', null);

        return !empty($items)
            ? collect($items)->keyBy($keyBy)->all()
            : null;

    }


//    /**
//     * @param string  $query
//     * @param array|bool $all
//     * @param string|null $type
//     * @param string|null $argument
//     *
//     * @return $this|mixed
//     */
//    public function query(string $query, $all = true, ?string $type = null, ?string $argument = null)
//    {
//        $this->reset();
//
//        if (is_array($all) || func_num_args() === 1) {
//            $params = explode('?', $query);
//            $newQuery = '';
//            foreach ($params as $key => $value) {
//                if (!empty($value)) {
//                    $newQuery .= $value . (isset($all[$key]) ? $this->escape($all[$key]) : '');
//                }
//            }
//            $this->query = $newQuery;
//            return $this;
//        }
//
//        $this->query = preg_replace('/\s\s+|\t\t+/', ' ', trim($query));
//
//        $str = false;
//        foreach (['select', 'optimize', 'check', 'repair', 'checksum', 'analyze'] as $value) {
//            if (stripos($this->query, $value) === 0) {
//                $str = true;
//                break;
//            }
//        }
//
//        $type = $this->getFetchType($type);
//        $cache = false;
//        if (!is_null($this->cache) && $type !== PDO::FETCH_CLASS) {
//            $cache = $this->cache->getCache($this->query, $type === PDO::FETCH_ASSOC);
//        }
//
//        if (!$cache && $str) {
//            $sql = $this->pdo->query($this->query);
//            if ($sql) {
//                $this->numRows = $sql->rowCount();
//                if ($this->numRows > 0) {
//                    if ($type === PDO::FETCH_CLASS) {
//                        $sql->setFetchMode($type, $argument);
//                    } else {
//                        $sql->setFetchMode($type);
//                    }
//
//                    $this->result = $all ? $sql->fetchAll() : $sql->fetch();
//                }
//
//                if (!is_null($this->cache) && $type !== PDO::FETCH_CLASS) {
//                    $this->cache->setCache($this->query, $this->result);
//                }
//                $this->cache = null;
//            } else {
//                $this->cache = null;
//                $this->error = $this->pdo->errorInfo()[2];
//                $this->error();
//            }
//        } else if ((!$cache && !$str) || ($cache && !$str)) {
//            $this->cache = null;
//            $this->result = $this->pdo->exec($this->query);
//
//            if ($this->result === false) {
//                $this->error = $this->pdo->errorInfo()[2];
//                $this->error();
//            }
//        } else {
//            $this->cache = null;
//            $this->result = $cache;
//            $this->numRows = is_array($this->result) ? count($this->result) : ($this->result == '' ? 0 : 1);
//        }
//
//        $this->queryCount++;
//
//
//        return $this->result;
//    }

    /**
     * Выполнение QUERY
     * @param string  $query
     * @param array|bool $params
     * @param string|null $fetchType
     * @param string|null $classArgument
     *
     * // Вашу компанию, продукт или ресурс мы рекламируем у себя на сайте и рассказываем нашим пользователям в социальных сетях, смс- рассылках, промо материалах и пр. Взамен Ваша организация для наших клиентов предоставляет скидки, ваучеры, промокоды и тд.
     *
     * @return $this|mixed
     */
    public function query(string $query, $params = true, ?string $fetchType = null, ?string $classArgument = null)
    {
        $this->reset();

        if (is_array($params) || func_num_args() === 1) {
            $this->query = $this->prepareQueryWithParams($query, $params);

            return $this;
        }

//        $this->log->write("Reason: " . $query);

        // Подготовка SQL-запроса с параметрами
        $this->query = $this->cleanQuery($query);

//        $this->log->write("Reason: " . $this->query);

        // Проверка типа запроса (выборка или изменения)
        $isSelectQuery = $this->isSelectQuery($this->query);

        // Получаем тип выборки и проверяем наличие кэша
        $fetchType = $this->determineFetchType($fetchType);
        $cachedResult = $this->getCachedResult($fetchType);

        // Если нет кэша, выполняем запрос
        if (!$cachedResult) {
            $this->result = $this->executeQuery($isSelectQuery, $fetchType, $classArgument, $params);
        } else {
            $this->result = $cachedResult;
            $this->numRows = $this->countRows($cachedResult);
        }

        $this->queryCount++;

        return $this->result;
    }


    /**
     * Подготовка SQL-запроса с параметрами
     *
     * @param  string  $query
     * @param $params
     *
     * @return string
     */
    private function prepareQueryWithParams(string $query, $params): string
    {
        $finalQuery = '';
        $segments = explode('?', $query);
        foreach ($segments as $key => $value) {
            if (!empty($value)) {
                $finalQuery .= $value . (isset($params[$key]) ? $this->escape($params[$key]) : '');
            }
        }

        return $finalQuery;
    }

    /**
     * TODO: протестировать
     * Очистка запроса от лишних пробелов
     *
     * @param  string  $query
     *
     *
     * Вашу компанию, продукт или ресурс мы рекламируем у себя на сайте и рассказываем нашим пользователям в социальных сетях, смс- рассылках, промо материалах и пр. Взамен Ваша организация для наших клиентов предоставляет скидки, ваучеры, промокоды и тд.
     *
     * We advertise your company, product or resource on our website and tell our users on social networks, SMS newsletters, promotional materials, etc. In return, your organization provides discounts, vouchers, promo codes, etc. for our customers.
     *
     * Biz sizning kompaniyangizni, mahsulotingizni yoki resursingizni veb - saytimizda reklama qilamiz va foydalanuvchilarimizga ijtimoiy tarmoqlarda, SMS-xabarnomalarda, reklama materiallarida va hokazolarda aytib beramiz, buning evaziga sizning tashkilotingiz mijozlarimiz uchun chegirmalar, vaucherlar, reklama kodlari va boshqalarni taqdim etadi.
     *
     * @return string
     */
    private function cleanQuery(string $query): string
    {

        // Разница между этими двумя выражениями preg_replace заключается в использовании модификатора `u`,
        // который гарантирует, что регулярное выражение корректно обрабатывает строки в кодировке UTF-8.

        // Это работает для сопоставления пробелов (\s) и символов табуляции (\t) в обычной строке ASCII.
        // может некорректно обрабатывать многобайтовые символы (например, в тексте, закодированном в UTF-8, например, буквы кириллицы в "русском"),
        // что может привести к проблемам с определенными кодировками символов.
//        return preg_replace('/\s\s+|\t+/', ' ', trim($query));

        // Check and convert to UTF-8 if necessary
//        if (!mb_check_encoding($query, 'UTF-8')) {
//            $query = mb_convert_encoding($query, 'UTF-8');
//
//        }

        // Это гарантирует, что ваша функция сможет безопасно и корректно обрабатывать многобайтовые символы, избегая повреждения символов или проблем с кодировкой.
        return trim(preg_replace('/\s+/u', ' ', $query));
    }

    /**
     * Определение, является ли запрос выборкой данных
     *
     * @param  string  $query
     *
     * @return bool
     */
    private function isSelectQuery(string $query): bool
    {
        foreach (['select', 'optimize', 'check', 'repair', 'checksum', 'analyze'] as $keyword) {
            if (stripos($query, $keyword) === 0) {
                return true;
            }
        }
        return false;
    }

    /**
     * Получение типа выборки
     *
     * @param  string|null  $fetchType
     *
     * @return int
     */
    private function determineFetchType(?string $fetchType): int
    {
        return $this->getFetchType($fetchType);
    }

    /**
     * Получение результата из кэша
     *
     * @param  int  $fetchType
     *
     * @return bool|null
     */
    private function getCachedResult(int $fetchType)
    {
        if ($this->cache && $fetchType !== PDO::FETCH_CLASS) {
            return $this->cache->getCache($this->query, $fetchType === PDO::FETCH_ASSOC);
        }

        return null;
    }

    /**
     * Выполнение SQL-запроса
     *
     * @param bool $isSelectQuery
     * @param int $fetchType
     * @param string|null $classArgument
     * @param $params
     *
     * @return false|mixed|null
     */
    private function executeQuery(bool $isSelectQuery, int $fetchType, ?string $classArgument, $params)
    {
        try {
            if ($isSelectQuery) {
                return $this->runSelectQuery($fetchType, $classArgument, $params);
            } else {
                return $this->pdo->exec($this->query);
            }
        } catch (PDOException $e) {
            $this->handleError($e->getMessage());
            return false;
        }
    }

    /**
     * Выполнение SELECT-запроса
     * @param int $fetchType
     * @param string|null $classArgument
     * @param $params
     *
     * @return false|mixed|null
     */
    private function runSelectQuery(int $fetchType, ?string $classArgument, $params)
    {
        $sql = $this->pdo->query($this->query);
        if ($sql) {
            $this->numRows = $sql->rowCount();
            if ($this->numRows > 0) {
                $fetchMode = ($fetchType === PDO::FETCH_CLASS) ? [$fetchType, $classArgument] : [$fetchType];
                $sql->setFetchMode(...$fetchMode);
                return $params ? $sql->fetchAll() : $sql->fetch();
            }
        }
        return [];
    }

    /**
     * Подсчет строк результата
     * @param $result
     *
     * @return int
     */
    private function countRows($result): int
    {
        return is_array($result) ? count($result) : ($result === '' ? 0 : 1);
    }

    /**
     * Обработка ошибок
     * @param string $errorMessage
     *
     * @return void
     */
    private function handleError(string $errorMessage)
    {
        $this->cache = null;
        $this->error = $errorMessage;
        $this->error(); // Логируем или обрабатываем ошибку
    }


    /**
     * @param $data
     *
     * @return string
     */
    public function escape($data)
    {
        return $data === null
            ? 'NULL'
            : (is_int($data) || is_float($data)
                ? $data
                : $this->pdo->quote($data));
    }

    /**
     * @param $time
     *
     * @return $this
     */
    public function cache($time): Connection
    {
        $this->cache = new Cache($this->_credentials['cacheDir'], $time);

        return $this;
    }

    /**
     * @return int
     */
    public function queryCount(): int
    {
        return $this->queryCount;
    }

    /**
     * @return string|null
     */
    public function getQuery(): ?string
    {
        if ($this->debug) {
            print "<!--\r\n";
            print "get query: ".$this->query."\r\n";
            print "-->";

            return $this->query;
        } else {
            return null;
        }
    }

    /**
     * @return void
     */
    public function __destruct()
    {
        $this->pdo = null;
    }

    /**
     * @return void
     */
    protected function reset(): void
    {
        $this->select = '*';
        $this->from = null;
        $this->where = null;
        $this->limit = null;
        $this->offset = null;
        $this->orderBy = null;
        $this->groupBy = null;
        $this->having = null;
        $this->join = null;
        $this->grouped = false;
        $this->numRows = 0;
        $this->insertId = null;
        $this->query = null;
        $this->error = null;
        $this->result = [];
        $this->transactionCount = 0;

    }

//    /**
//     * @return void
//     */
//    protected function resetIndexes(): void
//    {
//        $this->indexKey = null;
//        $this->valueKey = null;
//    }

    /**
     * @param $type
     * @return int
     */
    protected function getFetchType($type): int
    {
        return $type == 'class' ? PDO::FETCH_CLASS : ($type == 'array' ? PDO::FETCH_ASSOC : PDO::FETCH_OBJ);
    }

    /**
     * Optimize Selected fields for the query
     *
     * @param string $fields
     *
     * @return void
     */
    private function optimizeSelect(string $fields): void
    {
        $this->select = $this->select === '*' ? $fields : $this->select . ', ' . $fields;
    }

    private function optimizeSelectExists(string $fields): void
    {
        $this->selectExists = $this->selectExists === '*' ? $fields : $this->selectExists . ', ' . $fields;
    }


    /**
     * Парсинг строки таблицы для SQL-запроса.
     *
     * @param string $table Строка с именем таблицы и, возможно, алиасом.
     * @return string Имя таблицы с алиасом или без него.
     */
    private function parseTable(string $table): string
    {
        global $tbl;

        // Проверяем, содержит ли таблица алиас без разбора через preg_split
        if (strpos($table, ' ') !== false) {
            // Разбиваем строку на части (имя таблицы и алиас)
            $parts = preg_split('/\s+/', $table, 3); // Ограничиваем разбивку до 3 частей

            // Получаем имя таблицы
            $tableName = $tbl[trim($parts[0])] ?? trim($parts[0]);

            // Возвращаем таблицу с алиасом
            return  isset($parts[2]) ?  $this->_credentials['prefix'] . $tableName . ' AS ' . trim($parts[2]) :  $this->_credentials['prefix'] . $tableName;
        }

//        var_dump($this->_credentials);
        // Если алиаса нет, возвращаем таблицу
        return $this->_credentials['prefix'] . $tbl[trim($table)] ?? trim($table);
//        return 'ls_' . $tbl[trim($table)] ?? trim($table);
    }

//    /**
//     * @param $table
//     * @return mixed|string
//     */
//    private function parseTable($table)
//    {
//        global $tbl;
//
//        // Split the table string by spaces
//        $matches = explode(' ', $table);
//
//        // Check if matches array has elements
//        if (is_array($matches) && !empty($matches[0])) {
//            // Remove spaces and get the cleaned table name
//            $cleanedTable = str_replace(' ', '', $matches[0]);
//
//            // Check if there's an alias provided
//            if (isset($matches[1]) && isset($matches[2])) {
//                // Return table with alias
//                $alias = str_replace(' ', '', $matches[2]);
//                return $tbl[$cleanedTable] . ' AS ' . $alias;
//            }
//
//            // Return the table name without alias
//            return $tbl[$cleanedTable];
//        }
//
//        // Return the table if not matched
//        return $tbl[$table] ?? $table; // Return null if table is not found
//    }




    public function isColumn($columnName)
    {
        return $this->query("SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = '$this->databaseName' AND TABLE_NAME = '$this->from' AND COLUMN_NAME = '$columnName'");
    }
    public function createColumn($columnName, $datatype)
    {
        return $this->query("ALTER TABLE `$this->from` ADD COLUMN `$columnName` {$datatype}");
    }

    public function renameColumn($oldColumnName, $newColumnName, $schema)
    {
        return $this->query("ALTER TABLE `$this->from` RENAME COLUMN `$oldColumnName` to  `$newColumnName`");
    }

    public function changeColumnDatatype($columnName, $datatype)
    {
        return $this->query("ALTER TABLE `$this->from` MODIFY COLUMN `$columnName` {$datatype}");
    }

    public function removeColumn($columnName)
    {
        return $this->query("ALTER TABLE `$this->from` DROP COLUMN `$columnName`");
    }
}
