<?php

namespace Mnv\Core\Locale;

class LanguageManager
{
    private $languages = [];
    private array $L = [];
    private $adminLang;

    public function __construct($adminLang)
    {
        $this->adminLang = $adminLang;
        $this->loadLanguage($this->adminLang);
    }

    /**
     * Get the list of languages from the directory.
     *
     * @return array
     */
    public function getLanguages(): array
    {
        $languages = [];

        $dh = opendir(GLOBAL_ROOT . '/admin/languages');
        while ($item = readdir($dh)) {
            if ($item != '.' && $item != '..' && $item != '.DS_Store' && is_file(GLOBAL_ROOT . "/admin/languages/$item")) {
                $L = include(GLOBAL_ROOT . "/admin/languages/$item");
                $languages[$L['codeName']] = $this->text2unicode($L['langName'], $L['charset']);
            }
        }
        closedir($dh);

        return array_reverse($languages);
    }

    /**
     * Translate date format based on the language.
     *
     * @param string $date
     * @return string
     */
    public function langDate($date): string
    {
        $date = strtr($date, $this->L['dates']);
        return $this->array2unicode($date);
    }

    /**
     * Get the length of the language tag's corresponding value.
     *
     * @param string $tag
     * @param string|null $default
     * @return int
     */
    public function langLen(string $tag, ?string $default = ''): int
    {
        $text = $this->getTagVal($tag, $default);
        return strlen($text);
    }

    /**
     * Convert language tag's value to a JavaScript-safe format.
     *
     * @param string $tag
     * @param string|null $default
     * @return array|string
     */
    public function langJs(string $tag, ?string $default = '')
    {
        $text = $this->getTagVal($tag, $default);
        return $this->array2js($text);
    }

    /**
     * Get language text by tag.
     *
     * @param string $tag
     * @param string|null $default
     * @return array|string
     */
    public function lang(string $tag, ?string $default = '')
    {
        $text = $this->getTagVal($tag, $default);
        return $this->array2unicode($text);
    }

    /**
     * Replace language tags in a source string.
     *
     * @param string $source
     * @return string
     */
    public function replaceLanguageTags(string $source): string
    {
        if (!isset($this->L)) $this->loadLanguage($this->adminLang);
        return preg_replace_callback('/{([0-9a-zA-Z_\-:]+?)}/', [$this, 'replaceLangTagsCallback'], $source);
    }

    /**
     * Callback function for replacing language tags.
     *
     * @param array $matches
     * @return string
     */
    private function replaceLangTagsCallback($matches): string
    {
        $text = $this->getTagVal($matches[1], $matches[0]);
        return $this->array2unicode($text);
    }

    /**
     * Retrieve the value of a tag.
     *
     * @param string $tag
     * @param string $default
     * @return mixed
     */
    private function getTagVal($tag, $default)
    {
        $result = '';
        $keys = explode(':', $tag);
        switch (count($keys)) {
            case 1:
                $result = $this->L[$keys[0]] ?? $default;
                break;
            case 2:
                $result = $this->L[$keys[0]][$keys[1]] ?? $default;
                break;
            case 3:
                $result = $this->L[$keys[0]][$keys[1]][$keys[2]] ?? $default;
                break;
            case 4:
                $result = $this->L[$keys[0]][$keys[1]][$keys[2]][$keys[3]] ?? $default;
                break;
        }

        return $result;
    }

    /**
     * Convert an array to JavaScript-safe format.
     *
     * @param mixed $var
     * @return array|string
     */
    private function array2js($var)
    {
        if (is_array($var)) {
            return array_map([$this, 'array2js'], $var);
        }

        return $this->text2js($var, $this->L['charset']);
    }

    /**
     * Convert text to JavaScript-safe format.
     *
     * @param string $text
     * @param string $encoding
     * @return string
     */
    private function text2js($text, $encoding): string
    {
        return $this->text2unicode($text, $encoding);
    }

    /**
     * Convert an array to Unicode.
     *
     * @param mixed $var
     * @return array|string
     */
    private function array2unicode($var)
    {
        if (is_array($var)) {
            return array_map([$this, 'array2unicode'], $var);
        }

        return $this->text2unicode($var, $this->L['charset']);
    }

    /**
     * Convert text to Unicode.
     *
     * @param string $text
     * @param string $encoding
     * @return string
     */
    private function text2unicode($text, $encoding): string
    {
        mb_internal_encoding('UTF-8');
        return mb_convert_encoding($text, 'UTF-8', $encoding);
    }

    /**
     * Load the language file.
     *
     * @param string $language
     */
    private function loadLanguage(string $language)
    {
        if (is_file(GLOBAL_ROOT . '/admin/languages/' . $language . '.inc.php')) {
            $this->L = include(GLOBAL_ROOT . '/admin/languages/' . $language . '.inc.php');
        } else {
            $this->L = include(GLOBAL_ROOT . '/admin/languages/ru.inc.php');
        }
    }
}
