<?php

namespace Mnv\Core;

use Mnv\Http\Request;
use Mnv\Core\Smarty\Design;
use Mnv\Core\Utilities\Cookie\Cookie;

/**
 * Class Mnv
 * @package Mnv\Models
 */

class Mnv
{
    /** @var Design */
    public Design $smarty;

    /** @var array */
    protected array $config = [];

    /** @var string */
    protected string $action = '';

    /** @var string */
    protected string $module = '';

    /** @var string */
    protected string $status = '';
    /** @var string */
    protected string $query = '';

    protected ?array $filter = [];

    /** @var int */
    protected int $page = 1;

    /** @var int */
    protected int $limit = 20;

    /** @var string */
    public string $notice = '';

    /** @var array */
    public array $errors = [];

    /** @var array */
    public array $messages = [];

    /** @var array */
    public array $response = [];

    /** @var string */
    public string $buildQueryString = '';

    /** @var Request */
    public Request $request;

    /** @var string  */
    public string $adminUrl = '/admin';

    /** Mnv constructor. */
    public function __construct()
    {

        if (!empty(SITE_LANG)) {
            $this->adminUrl = '/admin/' . SITE_LANG;
        }

        $this->initializeRequest();
        $this->initializeConfig();
        $this->initializeSmarty();
        $this->initializeParams();
        $this->initializePaginationAndLimit();
        $this->initializeAction();
        $this->assignTemplateVariables();
    }

    /** Инициализация Request */
    protected function initializeRequest(): void
    {
        $this->request = Request::capture();
    }

    /** Инициализация конфигурации */
    protected function initializeConfig(): void
    {
        $this->config = ConfigManager::init()->config();
    }

    /** Инициализация Smarty */
    protected function initializeSmarty(): void
    {
        $this->smarty = Design::init()->design();
    }

    /** Инициализация параметров запроса */
    protected function initializeParams(): void
    {
        $this->module   = $this->request->get('mod','');
        $this->query    = (string) $this->request->get('query', '');
        $this->filter   = $this->request->get('filter');

        $this->buildQueryString($this->module);
    }

    /** Инициализация действия (action) */
    protected function initializeAction(): void
    {
        if ($this->request->request->has('action')) {
            $this->action = (string)$this->request->request->get('action', '');
        } elseif ($this->request->query->has('action')) {
            $this->action = (string)$this->request->query->get('action', '');
        }
    }

    /** Инициализация пагинации и лимита */
    protected function initializePaginationAndLimit(): void
    {
        $this->page = (int) $this->request->get('page', 1);
        $this->limit = $this->getLimitFromRequestOrCookie();
    }

    /** Получение лимита с учетом cookie */
    protected function getLimitFromRequestOrCookie(): int
    {
        $dsCookieName = 'limitSelector';
        $newLimit = (int)$this->request->get('limit', '');
        $dsCookie = json_decode($this->request->cookie($dsCookieName, '{}'), true);

        if ($newLimit) {
            $dsCookie['limit'] = $newLimit;
            Cookie::setcookie($dsCookieName, json_encode($dsCookie), time() + 60 * 60 * 24 * 180);
            return $newLimit;
        }

        return $dsCookie['limit'] ?? $this->limit;
    }

    /** Построение строки запроса */
    protected function buildQueryString(string $path, array $params = []): string
    {
        // Добавляем query-параметры, если они существуют
        if (!empty($this->query)) {
            $params['query'] = $this->query;
        }

        // Проверяем наличие и добавляем фильтры
        if (!empty($this->filter)) {
            $filters = array_filter([
                'filter[section]' => $this->filter['section'] ?? '',
                'filter[status]' => $this->filter['status'] ?? '',
                'filter[group]' => $this->filter['group'] ?? '',
                'filter[model]' => $this->filter['model'] ?? '',
            ], function ($value) {
                return $value !== null && $value !== '';
            });

            // Объединяем параметры и фильтры
            $params = array_merge($params, $filters);
        }

        $this->buildQueryString = !empty($params) ? '&' . http_build_query(array_filter($params), '', '&') : '';
        $this->buildQueryString = urldecode($this->buildQueryString);

        // Формируем итоговый URL
        $url = sprintf('%s/%s?%s', rtrim($this->adminUrl, '/'), trim($path), $this->buildQueryString);

        // Удаляем лишний `&` сразу после знака `?`, если он есть
        return str_replace('?&', '?', $url);
    }

    /** Назначение переменных для шаблона */
    protected function assignTemplateVariables(): void
    {
        $vars = [
            'adjustedNow'   => adjustTime(gmdate('Y-m-d H:i:s')),
            'page'          => $this->page,
            'limit'         => $this->limit,
            'query'         => $this->query,
            'filter'        => $this->filter,
            'buildQueryString' => $this->buildQueryString,
            'sortOrder'     => SortTable::init()->viewSortable($this->module),
            'errors'        => !empty($this->errors) ? $this->errors : null,
            'messages'      => !empty($this->messages) ? $this->messages : null,
            'action'        => !empty($this->action) ? $this->action : null,
        ];

        foreach ($vars as $key => $value) {
            if ($value !== null) {
                $this->smarty->assign($key, $value);
            }
        }
    }

}