# SecurityCsrfToken

`SecurityCsrfToken` — это простая, автономная библиотека защиты CSRF, написанная на PHP. Ее можно использовать 
для защиты форм от атак [Cross Site Request Forgery](http://en.wikipedia.org/wiki/Cross-site_request_forgery).

## Требования

* PHP 7.3+

Запустите `composer install`, а затем используйте как обычно:

```php
require 'vendor/autoload.php';

$sessionProvider = new Mnv\Core\Security\Csrf\SecurityCsrfSessionProvider();
$securityCsrfToken = new Mnv\Core\Security\Csrf\SecurityCsrfToken($sessionProvider);
```

## Использование

Чтобы использовать `SecurityCsrfToken`, сначала вам нужно сгенерировать токен:

```php
// хранение токена в сессии
$sessionProvider = new Mnv\Core\Security\Csrf\SecurityCsrfSessionProvider();
$securityCsrfToken = new Mnv\Core\Security\Csrf\SecurityCsrfToken($sessionProvider);

// или хранение токена в куках
$cookieProvider = new Mnv\Core\Security\Csrf\SecurityCsrfCookieProvider();
$securityCsrfToken = new Mnv\Core\Security\Csrf\SecurityCsrfToken($cookieProvider);

$token = $securityCsrfToken->generate('token');
```

Затем вы включаете этот токен во все создаваемые вами формы используя `smarty_function_security_csrf_token`, где использыуется код написаный выше:

```html
<form>
  
    // вернет готовый input с токеном
    {security_csrf_token input=true}

    // вызываем функцию и в assign указываем название переменной
    {security_csrf_token assign=token input=false}
    // выводим значение переменой {$token}
    <input type="hidden" name="token" value="{$token}" class="form-control d-none">

    // вернет токен
    <input type="hidden" name="token" value="{security_csrf_token input=false}" class="form-control d-none">

</form>
```

Затем, прежде чем приступать к обработке данных из форм, проверьте действительность токена:

```php
use Mnv\Core\Security\Csrf\Exceptions\InvalidCsrfTokenException;

try {
    $securityCsrfToken->check('token', $_POST['token']);
} catch(InvalidCsrfTokenException $e) {
    echo $e->getMessage();
}
```

## Истечение срока действия токена

Вы можете установить ограничение по времени для токенов, передав временной интервал (в секундах) в метод проверки. 
Токены старше временного интервала не будут действительны.

```php
// Example 1 hour expiration
$securityCsrfToken->check('token', $_POST['token'], 60 * 60);
```

## Многоразовые токены

Токены можно сделать многоразовыми, а не одноразовыми (полезно для запросов с большим объемом AJAX).

```php
// Make token reusable
$securityCsrfToken->check('token', $_POST['token'], null, true);
```

## Custom SessionProvider

Приложение может использовать стороннюю библиотеку для управления сеансами, 
или вы можете хранить токены где-то еще, а не в $_SESSION (как это делает `SecurityCsrfSessionProvider`). 
В этом случае вы можете создать пользовательский `SessionProvider`и использовать его при 
создании экземпляра SecurityCsrfToken.


```php
<?php

use Mnv\Core\Security\Csrf\Interfaces\SessionProvider;

class CustomSessionProvider implements SessionProvider
{
    /**
     * Get a session value.
     *
     * @param string $key
     * @return mixed
     */
    public function get(string $key)
    {
        // Верните ваши сохраненные данные
    }

    /**
     * Set a session value.
     *
     * @param string $key
     * @param mixed $value
     * @return void
     */
    public function set(string $key, $value)
    {
        // Сохраните ваши данные
    }

}
```

```php
$sessionProvider = new CustomSessionProvider();
$securityCsrfToken = new Mnv\Core\Security\Csrf\SecurityCsrfToken($sessionProvider);
```

