<?php

namespace Mnv\Core;

use Mnv\Core\Smarty\Design;

/**
 * Class responsible for handling the serialization and management of sections, trees
 * using internal arrays and external templates.
 */
class SerializationManager
{
    /** @var Design|\Smarty  */
    protected Design $smarty;

    /** @var array  */
    protected ?array $sections = [];

    /** @var array  */
    protected ?array $tree = [];

    /** @var string  */
    private string $SECTION_PATH;

    public function __construct()
    {
        global $smarty, $SECTIONS, $TREE;

        $this->smarty = $smarty;
        $this->sections = &$SECTIONS;
        $this->tree = &$TREE;
    }

    /**
     * Формирование serializations
     * @param  array  $buildLang
     *
     * @return bool
     */
    public function saveSerializations(array $buildLang = []): bool
    {
        $this->buildSectionArrays($buildLang);

        $tmpSmarty = Design::init()->design();

        $serializations = [
            'TREE'          => var_export($this->tree, true),
            'SECTIONS'      => var_export($this->sections, true)
        ];

        $this->smarty->assign('TREE', $this->tree);
        $this->smarty->assign('SECTIONS', $this->sections);

        $tmpSmarty->assign('serializations', $serializations);
        $tmpSmarty->default_modifiers = [];

        $siteRoot = empty($buildLang) ? SITE_ROOT : ($buildLang['isDefault'] ? GLOBAL_ROOT : GLOBAL_ROOT . '/' . $buildLang['codename']);

        $filePath = $siteRoot . '/includes/serializations.inc.php';
        $serializationsContent = $tmpSmarty->fetch(GLOBAL_ROOT . '/admin/templates/serializations.tpl', '', 'admin');

        return $this->writeFile($filePath, $serializationsContent);
    }


    public function buildSectionArrays(array $buildLang = []): bool
    {
        // если использовать полуный url с записью в базу
//       $this->SECTION_PATH = empty($buildLang) ? SITE_URL : ($buildLang['isDefault'] ? GLOBAL_URL : GLOBAL_URL.'/'.$buildLang['codename']);

        $this->SECTION_PATH = empty($buildLang) ? '' : ($buildLang['isDefault'] ? '' : '/' . $buildLang['codename']);

        $this->sections = connect('sections')->orderBy('parentId, sortOrder')->keyBy('sectionId');
        if (!empty($this->sections)) {
            foreach ($this->sections as $sectionId => &$section) {
                $section['parents'] = array_reverse($this->getAllParents($sectionId));
                $section['childLevels'] = 0;
                unset($section['content'], $section['addedBy'], $section['addedOn'], $section['modifiedBy'], $section['modifiedOn']);
                $section['level'] = count($section['parents']) + 1;
            }

            foreach ($this->sections as $sectionId => &$section) {
                $section['path'] = $this->SECTION_PATH;
                if (!empty($section['parents'])) {
                    $this->sections[$section['parents'][count($section['parents']) - 1]]['children'][] = $sectionId;
                    foreach ($section['parents'] as $parentId) {

                        /* if templateName is empty, look through parents for subTemplateName */
                        if (empty($section['templateName']) && !empty($this->sections[$parentId]['subTemplateName'])) {
                            $section['templateName'] = $this->sections[$sectionId]['templateName'] = $this->sections[$parentId]['subTemplateName'];
                        }

                        /* if subTemplateName is empty, look through parents for subTemplateName */
                        if (empty($section['subTemplateName']) && !empty($this->sections[$parentId]['subTemplateName'])) {
                            $section['subTemplateName'] = $this->sections[$sectionId]['subTemplateName'] = $this->sections[$parentId]['subTemplateName'];
                        }

                        /* if artTemplateName is empty, look through parents for artTemplateName */
                        if (empty($section['artTemplateName']) && !empty($this->sections[$parentId]['artTemplateName'])) {
                            $section['artTemplateName'] = $this->sections[$sectionId]['artTemplateName'] = $this->sections[$parentId]['artTemplateName'];
                        }

                        /* if isCached is 0 (use default), look through parents for subIsCached */
                        if ($section['isCached'] == 0 && $this->sections[$parentId]['subIsCached'] != 0) {
                            if ($section[$parentId]['subIsCached'] == 1) {
                                $section['isCached'] = $this->sections[$sectionId]['isCached'] = $this->sections[$parentId]['subIsCached'];
                                $section['cacheTime'] = $this->sections[$sectionId]['cacheTime'] = $this->sections[$parentId]['subCacheTime'];
                                $section['cachePeriod'] = $this->sections[$sectionId]['cachePeriod'] = $this->sections[$parentId]['subCachePeriod'];
                            } elseif ($this->sections[$parentId]['subIsCached'] == -1) {
                                $section['isCached'] = $this->sections[$sectionId]['isCached'] = -1;
                            }
                        }

                        /* if artIsCached is 0 (use default), look through parents for artIsCached */
                        if ($section['artIsCached'] == 0 && $this->sections[$parentId]['artIsCached'] != 0) {
                            if ($this->sections[$parentId]['artIsCached'] == 1) {
                                $section['artIsCached'] = $this->sections[$sectionId]['artIsCached'] = $this->sections[$parentId]['artIsCached'];
                                $section['artCacheTime'] = $this->sections[$sectionId]['artCacheTime'] = $this->sections[$parentId]['artCacheTime'];
                                $section['artCachePeriod'] = $this->sections[$sectionId]['artCachePeriod'] = $this->sections[$parentId]['artCachePeriod'];
                            } elseif ($this->sections[$parentId]['artIsCached'] == -1) {
                                $section['artIsCached'] = $this->sections[$sectionId]['artIsCached'] = -1;
                            }
                        }
                        /* if commentsEnabled is 0 (use default), look through parents for subCommentsEnabled */
                        if ($section['commentsEnabled'] == 0 && $this->sections[$parentId]['subCommentsEnabled'] != 0) {
                            if ($section[$parentId]['subCommentsEnabled'] == 1) {
                                $section['commentsEnabled'] = $this->sections[$sectionId]['commentsEnabled'] = $this->sections[$parentId]['subCommentsEnabled'];
                            } else if ($this->sections[$parentId]['subCommentsEnabled'] == -1) {
                                $section['commentsEnabled'] = $this->sections[$sectionId]['commentsEnabled'] = -1;
                            }
                        }


                        $this->sections[$parentId]['allChildren'][] = $sectionId;
                        $section['path'] .= '/' . $this->sections[$parentId]['fileName'];
                        if (empty($section['dir'])) {
                            $section['dir'] = $this->sections[$parentId]['fileName'];
                        } else {
                            $section['dir'] .= '/' . $this->sections[$parentId]['fileName'];
                        }
                        $this->updateChildLevels($sectionId, $parentId);
                    }
                }

                $this->finalizeSection($sectionId, $section);
            }

            $this->buildSectionTree();
        }

        return true;
    }

    protected function updateChildLevels(int $sectionId, int $parentId)
    {
        $section = &$this->sections[$sectionId];
        $parent = &$this->sections[$parentId];

        if ($parent['childLevels'] < ($section['level'] - $parent['level'])) {
            $parent['childLevels'] = $section['level'] - $parent['level'];
        }
    }

    protected function finalizeSection(int $sectionId, array &$section)
    {
        if (empty($section['dir'])) {
            $section['dir'] = $section['fileName'];
        } else {
            $section['dir'] .= '/' . $section['fileName'];
        }
        $section['path'] .= '/' . $section['fileName'];

        $tmpUrl = $this->generateSectionUrl($section);
        if ($section['url'] != $tmpUrl) {
            $section['url'] = $tmpUrl;

            connect()->table('sections')->where('sectionId', $sectionId)->update(['url' => $tmpUrl]);
            connect()->table('articles')->where('sectionId', $sectionId)->updateConcat("url", "CONCAT('" . $section['path'] . "', '/', fileName, '." . ConfigManager::getValue('file_extension') . "')");
        }
    }

    protected function generateSectionUrl(array &$section): string
    {
        if ($section['type'] == 'plain') {
            if ($section['fileName'] == 'index') {
                $tmpUrl =  $this->SECTION_PATH . '/';
            } else {
                $tmpUrl = $section['path'] . '/';
            }
        } else {
            $tmpUrl = $section['path'] . '/';
        }

        return $tmpUrl;
    }

    /**
     * @return void
     */

    protected function buildSectionTree()
    {
        $this->tree = $this->sections;
        $links = [];

        foreach ($this->tree as $sectionId => &$section) {
            $links[$sectionId] = &$section;
        }

        foreach ($this->tree as $sectionId => &$section) {
            $section['hasSubsections'] = false;

            if ($section['parentId'] > 0) {
                $parent = &$links[$section['parentId']];
                $parent['hasSubsections'] = true;
                $parent['subsections'][] = &$section;
                unset($this->tree[$sectionId]);
            }
        }

        usort($this->tree, [$this, 'sortBySortOrder']);
        $newSections = [];
        $this->sortAllSections($this->tree, $newSections);
        $this->sections = $newSections;
    }

    /**
     * @param  int  $sectionId
     * @param  array|null  $parents
     *
     * @return array
     */
    public function getAllParents(int $sectionId, ?array $parents = []): array
    {
        if ($this->sections[$sectionId]['parentId'] > 0) {
            $parents[] = $this->sections[$sectionId]['parentId'];
            $parents = $this->getAllParents($this->sections[$sectionId]['parentId'], $parents);
        }
        return $parents;
    }

    /**
     * @param  array  $sections
     * @param $newSections
     *
     * @return void
     */
    protected function sortAllSections(array $sections, &$newSections)
    {
        foreach ($sections as $section) {
            $sectionId = $section['sectionId'];
            $newSections[$sectionId] = $this->sections[$sectionId];

            if (!empty($section['subsections'])) {
                $this->sortAllSections($section['subsections'], $newSections);
            }
        }
    }

    /**
     * Сортировка
     * @param  array  $a
     * @param  array  $b
     *
     * @return int
     */
    public function sortBySortOrder(array &$a, array &$b): int
    {
        if (!empty($a['subsections'])) {
            usort($a['subsections'], [$this, 'sortBySortOrder']);
        }
        if (!empty($b['subsections'])) {
            usort($b['subsections'], [$this, 'sortBySortOrder']);
        }

        return $a['sortOrder'] < $b['sortOrder'] ? -1 : 1;
    }


    /** ******************* htaccess ******************** */

    /**
     * Формирование htaccess
     * @param array $buildLang
     * @return bool
     */
    public function writeHtaccess(array $buildLang = []): bool
    {
        // Сгенерируем уникальный хэш для перезаписи
        $rewrite = md5(rand() * time());

        // Инициализируйте движок шаблонов Smarty с необходимыми конфигурациями
        $tmpSmarty = Design::init()->design();
        $this->setupSmarty($tmpSmarty, 'admin-' . ConfigManager::getValue('admin_language'));

        // Назначьте существующие переменные шаблона из $this->smarty временным переменным $tmpSmarty
        $tmpSmarty->assign($this->smarty->getTemplateVars());

        // Назначаем переменную rewrite
        $tmpSmarty->assign('rewrite', $rewrite);

        // Найти идентификатор `$homePageId`
        $homePageId = $this->getHomePageId();
        if (!empty($homePageId)) {
            $tmpSmarty->assign('homePageId', $homePageId);
        }

        // Определяем корень сайта и шаблон на основе языковых настроек
        $siteRoot = empty($buildLang) ? SITE_ROOT : ($buildLang['isDefault'] ? GLOBAL_ROOT : GLOBAL_ROOT . '/' . $buildLang['codename']);
        $htaccessTPL = $this->getHtaccessTemplate($buildLang);

        // Извлекаем и записываем файл .htaccess
        $htaccessContent = $tmpSmarty->fetch(GLOBAL_ROOT . '/admin/templates/' . $htaccessTPL);
        $filePath = $siteRoot . '/.htaccess';
        if ($this->writeFile($filePath, $htaccessContent)) {
            // Сохраните начальный код `rewrite` в настройках
            connect('settings')->replace([
                'codename' => 'rewrite',
                'value' => $rewrite
            ]);

            unset($tmpSmarty);

            return true;
        }

        return false;
    }

    private function setupSmarty($smarty, $compileId)
    {
        $smarty->setTemplateDir($this->smarty->defaultAdminTemplateDir());
        $smarty->setCacheDir($this->smarty->tmpPath() . 'cache');
        $smarty->setCompileDir($this->smarty->tmpPath() . 'compile');
        $smarty->setCompileId($compileId);
        $smarty->default_modifiers = ['escape'];
    }

    private function getHomePageId(): ?int
    {
        $homePageId = null;
        foreach ($this->sections as $section) {
            if (empty($homePageId) || $section['fileName'] == 'index') {
                $homePageId = $section['sectionId'];
            }
        }
        return $homePageId;
    }

    private function getHtaccessTemplate($buildLang): string
    {
        if (empty($buildLang)) {
            return SITE_LANG == '' ? 'htaccess.tpl' : 'htaccess-ml.tpl';
        } else {
            $isDefault = $buildLang['isDefault'];
            return $isDefault ? 'htaccess.tpl' : 'htaccess-ml.tpl';
        }
    }



    private function writeFile($filePath, $content): bool
    {
        if ($fh = fopen($filePath, 'w')) {
            fwrite($fh, $content);
            fclose($fh);
            return true;
        }
        return false;
    }

}