<?php

namespace Mnv\Core\Smarty;

use Closure;
use Smarty;
use SmartyException;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
class SmartyPlugins
{

    /**
     * Регистрация всех плагинов, включая модификаторы и функции
     *
     * @param Smarty $smarty
     * @return void
     * @throws SmartyException
     */
    public static function register(Smarty $smarty): void
    {

        // Модификаторы
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'in_array', self::inArray());
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'utf8_encode', [self::class, 'smartyModifierUtf8Encode']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'upper',       [self::class, 'smartyModifierUpper']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'array_first', [self::class, 'smartyModifierArrayFirst']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'element',		[self::class, 'smartyModifierElement']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'remove_element',	[self::class, 'smartyModifierRemoveElement']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'break', 	    [self::class, 'smartyModifierBreak']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'page_break', 	[self::class, 'smartyModifierPageBreak']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'substr', 		 [self::class, 'smartyModifierSubstr']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'array_chunk',  [self::class, 'smartyModifierArrayChunk']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'fsize_format', [self::class, 'smartyModifierFileSizeFormat']);
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'plural',       [self::class, 'smartyModifierPlural']);


        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'preg_quote', function($string) {
            return preg_quote($string, '/');
        });


        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'strtotime', function ($string) {
            return strtotime($string);

        });

        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'format_time', function ($string, $format = 'r') {
            if ($string === '' || $string === null) {
                $string = time(); // Используем текущую метку времени
            }

            return date($format, is_numeric($string) ? $string : strtotime($string));
        });

        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'first', function ($value) {
            if (is_array($value)) {
                return reset($value);
            }
            if (is_string($value)) {
                return $value[0] ?? '';
            }
            if (is_object($value)) {
                // Для объекта: берём первый элемент при итерации
                foreach ($value as $item) {
                    return $item;
                }
                return null; // Если объект пустой
            }
            return null;
        });


        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'print_r', function ($value) {
            if (empty($value)) {
                return 'Пустое значение';
            }

            if (is_array($value) || is_object($value)) {
                return print_r($value, true);
            }

            return (string)$value;
        });

        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'var_dump', function ($value) {
            // Используем буферизацию вывода
            ob_start();

            var_dump($value); // Выводим данные через var_dump

            // Получаем буферизированный вывод как строку
            $output = ob_get_clean();

            // Возвращаем вывод модификатору
            return $output;
        });

        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'extract_int', function ($value) {
            if (preg_match_all('/\d+/', $value, $matches)) {
                return (int) implode('', $matches[0]); // Объединяем найденные числа и преобразуем в целое
            }
            return 0; // Возвращаем 0, если цифры не найдены
        });

        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'intval', function ($value) {
            if (is_numeric($value)) {
                return intval($value); // Преобразование только для числовых данных
            }
            return 0; // Возвращаем 0 для некорректных значений
        });


        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'ceil', function ($value) {
            // Проверяем, является ли значение числом
            if (is_numeric($value)) {
                return ceil($value);
            }
            // Для некорректных данных возвращаем 0 или любое другое значение по умолчанию
            return 0;
        });



        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'unset_key', function ($array, $key) {
            if (is_array($array) && array_key_exists($key, $array)) {
                unset($array[$key]);
            }
            return $array;
        });

        /**
         *  Регистрируем PHP-функцию explode как модификатор
         */
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'explode', function ($string, $delimiter = ',') {
            if (!is_string($string) || !is_string($delimiter)) {
                return [];
            }

            return explode($delimiter, $string);
        });

        /**
         * Registers a custom Smarty modifier "slice" for slicing arrays or strings.
         *
         * @example {assign var="galleryFirst" value=$section.gallery|slice:0:2} - показать от 0 до 2
         * @example {assign var="galleryRest" value=$section.gallery|slice:2} - показать от 2
         */
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, "slice", function ($data, $start, $length = null) {
            if (is_array($data)) {
                return array_slice($data, (int) $start, $length !== null ? (int) $length : null);
            }

            if (is_string($data)) {
                return mb_substr($data, (int) $start, $length !== null ? (int) $length : null); // Multibyte-safe
            }

            return $data; // Возвращаем неизмененные входные данные для неподдерживаемых типов
        });


        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'string_format',  function($string, $format) {
            return sprintf($format, $string);
        });

        /**
         * @param $phone
         * @return array|string|string[]|null
         */
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'format_phone',  function($phone) {
            return \Mnv\Core\Utilities\Mask::init()->phone($phone);
        });

        /**
         * @param $phone
         * @return array|string|string[]|null
         */
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'format_phone_old',  function($phone) {
            $phone = preg_replace('/[^0-9]/', '', $phone);
            return preg_replace('/([0-9]{3})([0-9]{2})([0-9]{3})([0-9]{2})([0-9]{2})/', '+$1 ($2) $3-$4-$5', $phone);
        });


        /**
         * @param $phone
         * @return string
         */
        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'strip_phone', function($phone) {
            $phone = preg_replace('/[^0-9]/', '', $phone);
            return '+' . $phone;
        });


        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'sizeof',  function($value)
        {
            if (is_array($value) || (is_object($value) && $value instanceof \Countable)) {
                return count($value);
            }

            return 0;
        });


        $smarty->registerPlugin(Smarty::PLUGIN_MODIFIER, 'htmlspecialchars', function ($string)
        {
            return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
        });

        // Блоки
        self::registerBlockPlugins($smarty);

    }

    /**
     * Регистрация блоков для Smarty
     *
     * @param Smarty $smarty
     * @return void
     */
    private static function registerBlockPlugins(Smarty $smarty): void
    {
        try {
            $smarty->registerPlugin(Smarty::PLUGIN_BLOCK, 'nocache', self::nocacheBlock(), false, '');
        } catch (SmartyException $e) {
            // Обработка исключений, если регистрация не удалась
            error_log($e->getMessage());
        }
    }

    /**
     * Блок "nocache"
     *
     * @return \Closure
     */
    private static function nocacheBlock(): \Closure
    {
        return function ($params, $content, &$smarty) {
            return $content;
        };
    }

    /**
     * Модификатор "in_array" для Smarty
     *
     * @example {$value|in_array:$array}
     *
     * @return \Closure
     */
    private static function inArray(): \Closure
    {
        return function ($needle, $haystack): bool {
            if (!is_array($haystack) && !empty($haystack)) {
                $haystack = [$haystack];
            }

            return is_array($haystack) && in_array($needle, $haystack);
        };
    }


    public static function smartyModifierUtf8Encode(?string $string): string
    {
        if (empty($string)) {
            return '';
        }

        if (mb_detect_encoding($string, 'UTF-8', true) === 'UTF-8') {
            return $string; // Строка уже в кодировке UTF-8
        }

        return mb_convert_encoding($string, 'UTF-8', 'auto');
    }


    public static function smartyModifierUpper(?string $string): ?string
    {
        if (empty($string)) {
            return '';
        }

        // Проверяет, содержит ли строка кириллические символы
        if (Smarty::$_MBSTRING) {
            return mb_strtoupper($string, addslashes(Smarty::$_CHARSET));
        }

        // Для некирилловских символов используйте `strtoupper` по умолчанию
        return strtoupper($string);

    }

    public static function smartyModifierArrayFirst($array, $key,  callable $callback = null, $default = null)
    {
        if (is_null($callback)) {
            if (empty($array)) {
                return $default instanceof Closure ? $default() : $default;
            }

//            print_r($array);
            foreach ($array as $item) {
                return $item[$key];
            }
        }

        foreach ($array as $key => $value) {
            if ($callback($value, $key)) {
                return $value;
            }
        }

        return $default instanceof Closure ? $default() : $default;
    }

    /**
     * @param array|null $params
     * @param $key
     * @param $value
     * @param $as
     * @return false|mixed|null
     */
    public static function smartyModifierElement(?array $params, $key, $value, $as = null)
    {
        if (!is_array($params)) {
            return false;
        }

        $result = null;
        foreach ($params as $param) {
            if (isset($param[$key])) {
                if ($param[$key] !== $value) {
                    continue;
                }
                if (!empty($as)) {
                    $result = $param[$as];
                } else {
                    $result = $param;
                }
            }
        }

        return $result;
    }


    /**
     * @param array|null $array
     * @param $value
     * @return false|mixed|null
     */
    public static function smartyModifierRemoveElement(?array $array, $value)
    {
        if (!is_array($array)) {
            return false;
        }

        unset($array[array_search($value, $array, true)]);
        return implode(',', $array);
    }



    public static function smartyModifierBreak($content): array
    {
        if (empty($content)) {
            return [];
        }

        if (stripos($content, '{PAGEBREAK}') !== false) {
            $content = explode('{PAGEBREAK}', $content);
            return str_replace('{PAGEBREAK}', '', $content);
        }

        return [$content];

    }

    /**
     * @param $content
     * @param null $key
     * @return array|false|mixed|string|string[]
     */
    public static function smartyModifierPageBreak($content, $key = NULL)
    {
        if (empty($content)) {
            return $content; // Возвращаем контент как есть, если он пустой
        }

        // Если передан ключ и содержится разделитель {PAGEBREAK}, работаем с ним
        if (stripos($content, '{PAGEBREAK}') !== false) {
            if ($key === null) {
                // Удаляем все упоминания {PAGEBREAK} и возвращаем контент
                return str_ireplace('{PAGEBREAK}', '', $content);
            }

            // Разделяем контент на части
            $contentParts = array_map('trim', explode('{PAGEBREAK}', $content));

            // Если передан ключ $key, возвращаем соответствующую часть или пустую строку
            return $contentParts[$key] ?? '';
        }

        // Возвращаем исходный контент, если разделителя нет
        return $content;

    }


    public static function smartyModifierSubstr($string, $start, $length): ?string
    {
        if (is_null($string)) {
            return null;
        }

        $start = (int)$start;
        $start = $start <= strlen($string) && $start >= -strlen($string) ? $start : 0;
        $length = $length ? (int)$length : null;

        return substr($string, $start, $length);
    }

    public static function smartyModifierArrayChunk(array $array, int $length, bool $preserve_keys = false): array
    {
        // Проверка на пустой массив или некорректную длину
        if (empty($array) || $length <= 0) {
            return [];
        }

        // Используем array_chunk для разбиения массива
        return array_chunk($array, $length, $preserve_keys);
    }

    public static function smartyModifierFileSizeFormat($size, $format = '', $precision = 2, $dec_point = ".", $thousands_sep = ","): string
    {
        $format = strtoupper($format);

        static $sizes = array();

        if (!count($sizes)) {
            $b = 1024;
            $sizes["B"]        =    1;
            $sizes["KB"]    =    $sizes["B"]  * $b;
            $sizes["MB"]    =    $sizes["KB"] * $b;
            $sizes["GB"]    =    $sizes["MB"] * $b;
            $sizes["TB"]    =    $sizes["GB"] * $b;

            $sizes = array_reverse($sizes,true);
        }

        //~ get "human" filesize
        foreach($sizes AS $unit => $bytes) {
            if($size > $bytes || $unit === $format) {
                //~ return formatted size
                return number_format($size / $bytes, $precision, $dec_point, $thousands_sep)." ".$unit;
            } //~ end if
        } //~ end foreach

        return '';
    }

    /**
     * @param $number
     * @param $singular
     * @param $plural1
     * @param null $plural2
     * @return mixed
     */
    public static function smartyModifierPlural($number, $singular, $plural1, $plural2 = null)
    {
        $number = abs($number);
        if(!empty($plural2)) {
            $p1 = $number%10;
            $p2 = $number%100;
            if ($number === 0) {
                return $plural1;
            }
            if ($p1 === 1 && !($p2 >= 11 && $p2 <= 19)) {
                return $singular;
            }

            if ($p1 >= 2 && $p1 <= 4 && !($p2 >= 11 && $p2 <= 19)) {
                return $plural2;
            }

            return $plural1;
        }

        if ($number === 1) {
            return $singular;
        }

        return $plural1;
    }
}