<?php


namespace Mnv\Core\Uploads;

use Mnv\Core\ConfigManager;
use Mnv\Core\Filesystem\Filesystem;
use Mnv\Core\Uploads\Exceptions\InvalidParamException;
use Mnv\Core\Utilities\GUID;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

/**
 * Class AbstractImagineGenerator
 * @package Mnv\Core\Uploads
 */
abstract class AbstractGenerator extends Filesystem
{

    /** @var array  */
    protected array $sizes = array('large', 'medium', 'small');

    /** @var string  */
    public string $realPath;

    /** @var string */
    public string $path;

    /** @var string  */
    public string $uuid;

    /** @var int */
    public int $managerId;

    /** @var SymfonyUploadedFile */
    protected SymfonyUploadedFile $file;

    /** @var string */
    protected string $fileName;

    /** @var string  */
    protected string $driver = 'gd';

    /** @var string  */
    protected string $extension;

    /** @var array */
    protected array $options = [];

    /** @var array */
    protected array $fontOptions = [];

    /**
     * @var int|mixed
     */
    protected int $imageLibrary;

    /**
     * @var int|mixed
     */
    protected int $thumbnailQuality;

    /** @var string  */
    protected string $fontWatermark = GLOBAL_ROOT . '/admin/assets/fonts/verdana.ttf';

    /**
     * AbstractImagineGenerator constructor.
     *
     * @param string      $realPath  Реальный путь к файлу.
     * @param string      $path      Путь для хранения файла.
     * @param string|null $uuid      Уникальный идентификатор файла (UUID).
     * @param int         $managerId ID менеджера, создающего файл.
     */
    public function __construct(string $realPath, string $path, ?string $uuid, int $managerId)
    {
        $this->realPath     = $realPath;
        $this->path         = $path;
        $this->uuid         = $uuid;
        $this->managerId    = $managerId;

        $this->thumbnailQuality = ConfigManager::getValue('thumbnail_quality') ?? 90;
        $this->imageLibrary = ConfigManager::getValue('image_library') ?? 0;
    }

    abstract public function init(SymfonyUploadedFile $file);

    /**
     * если включена конвертация файла в `webp` то нам нужно знать,
     * если это `gif` то нельзя конвертировать в `webp`,
     * для остальных расширений разрешена конвертация в `webp`
     * @throws InvalidParamException
     */
    protected function setGeneratorExtension(string $originalExtension): void
    {
        // Приводим расширение к нижнему регистру
        $originalExtension = strtolower(trim($originalExtension));

        // Проверка корректности входного расширения
        if (empty($originalExtension)) {
            throw new InvalidParamException('File extension cannot be empty.');
        }

        // Проверяем необходимость преобразования в WebP или сохраняем оригинальное расширение
        if ($originalExtension === 'gif' || ConfigManager::getValue('force_webp') === 0) {
            $this->extension = $originalExtension;
        } else {
            $this->extension = 'webp';
        }
    }

    protected function generateFileName(string $originalName)
    {
        if (ConfigManager::getValue('keep_original_image_name')) {
            $this->fileName = pregGetValue('/([^\/]+)\.[^\.]+$/', $originalName);
        }

        /** новое название файла uuid */
        $this->fileName = !empty($this->uuid) ? $this->uuid : GUID::Format(GUID::Create(), false, '-');
    }

    /**
     * сбор опций обработки `watermark` $this->fontOptions = [
     * 'Размер шрифта (size)'        => 'watermark_font',
     * 'Цвет шрифта (color)'         => 'watermarkColor',
     * 'Прозрачность шрифта (alpha)' => 'watermark_opacity',  // пока не работает
     * 'Поворот текста (angle)'      => 'watermark_rotate'
     * ];
     */
    protected function collectFontOptions()
    {
        $this->fontOptions = [
            'size' => ConfigManager::getValue('watermark_font'),
            'color' => [
                'light' => ConfigManager::getValue('watermark_color_light'),
                'dark' => ConfigManager::getValue('watermark_color_dark')
            ],
            'alpha' => (int)ConfigManager::getValue('watermark_opacity'),
            'angle' => ConfigManager::getValue('watermark_rotate'),
            'margin' => 20,
        ];
    }


    /**
     * @param array $file
     * @return bool|int|string|null
     */
    public function saveFile(array $file)
    {
        if (empty($file)) {
            return false;
        }

        $insertFile = array_filter([
            'directory' => $this->basename($this->realPath),
            'path'      => '/uploads/' . $this->path,
            'fileName'  => $file['name'],
            'size'      => $file['size'],
            'mimeType'  => $file['mimeType'] ?? null,
            'addedBy'   => $this->managerId ?: 0,
            'addedOn'   => gmdate('Y-m-d H:i:s'),
        ]);

        if (!empty($insertFile)) {
            return connect('files')->insert($insertFile) ?: false;
        }

        return false;
    }

    /**
     * Формирует размеры для обработки файла.
     *
     * @param string|int $size Размеры в формате "ширинаxвысота" или фиксированное значение.
     * @return array Массив с шириной и высотой [width, height].
     * @throws \InvalidArgumentException Если формат размеров некорректен.
     */
    protected function autoSize($size): array
    {
        // Если передан строковый формат "WxH"
        if (is_string($size) && strpos($size, 'x') !== false) {
            $sizes = explode('x', $size);

            // Проверяем, что значения числовые
            if (count($sizes) !== 2 || !ctype_digit($sizes[0]) || !ctype_digit($sizes[1])) {
                throw new \InvalidArgumentException("Invalid size format: '{$size}'. Expected format: 'widthxheight'.");
            }

            [$width, $height] = array_map('intval', $sizes);
        } elseif (is_numeric($size)) {
            // Если передано одно число, предполагается квадратное соотношение
            $width = $height = intval($size);
        } else {
            // Если формат не распознан
            throw new \InvalidArgumentException("Invalid size provided. Expected integer or string in format 'widthxheight'.");
        }

        return [$width, $height];
    }

}