<?php

namespace Mnv\Core\Uploads;

use Mnv\Core\ConfigManager;
use Mnv\Core\Uploads\Exceptions\InvalidParamException;
use Mnv\Core\Utilities\GUID;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

use Intervention\Image\Image;
use Intervention\Image\ImageManager;


/**
 * Class ImageGenerator
 * @package Mnv\Core\Uploads
 */
class ImageGenerator extends AbstractGenerator implements IGenerator
{

    /** @var Image */
    private Image $image;

    /** @var ImageManager  */
    private ImageManager $manager;

    /**
     * Конструктор класса.
     *
     * @param string      $realPath  Реальный путь до файла.
     * @param string      $path      Путь для сохранения файла.
     * @param string|null $uuid      Уникальный идентификатор файла (UUID).
     * @param int         $managerId ID менеджера, создавшего файл.
     */
    public function __construct(string $realPath, string $path, ?string $uuid, int $managerId)
    {
        // Вызываем конструктор родительского класса
        parent::__construct($realPath,  $path, $uuid, $managerId);

        // Установка драйвера
        $this->setDriver();
    }

    /**
     * @param SymfonyUploadedFile $file
     *
     * @return ImageGenerator
     * @throws InvalidParamException
     */
    public function init(SymfonyUploadedFile $file): ImageGenerator
    {
        if (!$file->isValid() || !is_readable($file->getPathname())) {
            throw new InvalidParamException('The uploaded file is invalid or cannot be read.');
        }

        $this->file = $file;

        $this->setGeneratorExtension($file->getClientOriginalExtension());

        // Генерация имени нового файла
        $this->generateFileName($file->getClientOriginalName());

        // Сбор опций для шрифтов и водяных знаков
        $this->collectFontOptions();

        try {
            // Создаем изображение с использованием ImageManager
            $this->image = $this->manager->make($this->file->getPathname());
        } catch (\Exception $e) {
            throw new InvalidParamException('Failed to initialize image.', 0, $e);
        }

        return $this;
    }

    /**
     * @return bool|int|string|null
     * @throws InvalidParamException
     */
    public function save()
    {
        $images = [
            'name'      => "{$this->fileName}.{$this->extension}",
            'size'      => $this->file->getSize() ?? null,
            'mimeType'  => ($this->extension == 'webp') ? 'image/webp' : $this->file->getClientMimeType(),
        ];


        $this->resizeImage($this->image->getWidth(), $this->image->getHeight(),$this->realPath . $this->fileName . '.' . $this->extension);

        foreach ($this->sizes as $size) {
            $configKey = "{$size}_thumb";
            $configValue = ConfigManager::getValue($configKey);

            if (empty($configValue)) {
                throw new InvalidParamException("Missing or invalid configuration for key: {$configKey}");
            }

            $sizes = $this->autoSize($configValue);
            $this->resizeImage($sizes[0], $sizes[1], $this->realPath . $size . '/' . $this->fileName . '.' . $this->extension);
        }

        return $this->saveFile($images);
    }

    /**
     * обработка и сохранение размеров
     *
     * @param int $width
     * @param int $height
     * @param string $path
     * @throws InvalidParamException
     */
    public function resizeImage(int $width, int $height, string $path)
    {
        // Проверяем существование директории
        $directory = dirname($path);
        if (!is_dir($directory) && !mkdir($directory, 0777, true)) {
            throw new InvalidParamException("Unable to create directory: {$directory}");
        }

        if (ConfigManager::getValue('allow_watermark') && $this->extension != 'gif') {
            $this->generateWatermark();
        }

        $this->image->resize($width, $height, function ($constraint) {
                $constraint->aspectRatio();
                $constraint->upsize();
            })->save($path, $this->thumbnailQuality, $this->extension);
    }

    private function generateWatermark()
    {
        $this->image->text(ConfigManager::getValue('watermark_text'), $this->image->width() / 2, $this->image->height() / 2, function ($font) {
            $font->file($this->fontWatermark);
            $font->size($this->fontOptions['size']);
            $font->color($this->fontOptions['color']['light']);
            $font->align('center');
            $font->valign('top');
            $font->angle($this->fontOptions['angle']);
        });
    }

    /**
     * Sets the image processing driver for the ImageManager instance.
     *
     * @return void
     * @throws \RuntimeException If an unsupported image driver is specified.
     */
    private function setDriver(): void
    {
        $driver = 'gd'; // Устанавливаем драйвер по умолчанию

        if ($this->imageLibrary != 2) {
            if (extension_loaded('imagick') && class_exists('Imagick')) {
                $driver = 'imagick';

                // Проверяем поддержку формата WEBP
                if (!\Imagick::queryFormats('WEBP') && function_exists('imagewebp') && $this->imageLibrary != 1) {
                    $driver = 'gd';
                }
            }
        }

        if (!in_array($driver, ['gd', 'imagick'])) {
            throw new \RuntimeException('Unsupported image driver.');
        }

        $this->manager = new ImageManager(['driver' => $driver]);
    }


}