<?php

namespace Mnv\Models;


use Mnv\Core\Bot\Telegram;
use Mnv\Core\Collections\Contracts\Filesystem\FileNotFoundException;
use Mnv\Core\Filesystem\Filesystem;
use Mnv\Core\Test\Log;
use Mnv\Http\Request;

/**
 * Class TelegramNotification
 *
 * Handles Telegram notification functionalities including configuration management,
 * reading and writing configuration files, setting up webhooks, reading log files,
 * and retrieving template variables.
 */
class TelegramNotification
{

    /** @var string  */
    private string $version = '1.0.1';

    /** @var Filesystem  */
    private Filesystem $filesystem;

    /** @var bool  */
    private bool $ssl = false;

    /** @var array|null */
    public ?array $data = null;

    /** @var string|null */
    public ?string $key;

    /** @var array|string[]  */
    public array $logs = [
        0 => 'Выключены',
        1 => 'Только ошибки',
        2 => 'Полные (все запросы: успешные и ошибочные + тест уведомлений)',
    ];


    /** @var array|string[]  */
    public array $modal_headers = [
        'tab_template'          => 'Шаблоны уведомлений',
        'tab_users'             => 'Настройки пользователей',
        'tab_logs'              => 'Логи',
        'help_module'           => 'Инструкция по настройке модуля',
        'help_bot_telegram'     => 'Создать бота в Telegram',
        'help_trim_messages'    => 'Длинное сообщение',
        'help_timeout'          => 'Таймаут соединения',
        'help_proxy'            => 'Прокси',
        'help_id_telegram'      => 'Как получить ID пользователя в Telegram',
        'help_id_viber'         => 'Как получить ID пользователя в Viber',
        'help_log'              => 'Лог',
        'help_faq'              => 'Часто Задаваемые Вопросы',
        'help_support'          => 'Поддержка',
        'error_token'           => 'Неправильный Bot Token или недоступен мессенджер!',
        'error_warning'         => 'Внимание: Ваш  файл ошибок %s имеет размер %s!',

        'entry_get_token_telegram'      => 'Как получить Telegram Bot Ключ (токен)',
    ];


    /**
     * @param Request $request
     */
    public function __construct(Request $request)
    {
        $this->filesystem = new Filesystem();

        $this->key = $request->get('key');
        $this->data = $request->get('telegram');
        $this->data['telegram_user_data'] = $request->get('telegram_user_data');
        $this->data['telegram_proxy_data'] = $request->get('telegram_proxy_data');
        $this->data['telegram_user_groupdata'] = $request->get('telegram_user_groupdata');
        $this->data['telegram_langdata'] = $request->get('telegram_langdata');

        $this->ssl = $request->server('HTTPS') === 'on' || $request->server('HTTPS') === '1';

    }

    /**
     * @return void
     */
    public function readTelegramArray(): void
    {
        $path = GLOBAL_ROOT . '/includes/config.telegram.inc.php';
        if (!file_exists($path)) {
            throw new \RuntimeException("Config file not found: $path");
        }

        $this->data = require $path;
    }

    /**
     * @return void
     */
    public function buildTelegramArray(): void
    {
        $data = "<?php\r\n";
        $data .= 'return ' . var_export($this->data, true) . ';';

        $this->filesystem->put(GLOBAL_ROOT . '/includes/config.telegram.inc.php', $data);
    }


    /**
     * @param $key
     * @param $timeout
     * @return array
     */
    public function set_webhook($key, $timeout): array
    {
        $telegram = new Telegram($key, $timeout);
        $telegram->setLog( new Log('telegram.log'), 2);

        if ($this->ssl) {
            $telegram->setWebhook(GLOBAL_URL . '/api-telegram');
        }

        $response = $telegram->getWebhookInfo();

        if (!$response) {
            if (($this->data['proxy'] == '1') && (!empty($this->data['proxydata']))) {
                $results = explode(';', $this->data['proxydata']);
                foreach ($results as $proxy) {
                    if (!$response) {
                        $telegram->setProxy($proxy);
                        $response = $telegram->getBotInfo();

                        if ($response) {
                            $json['webhook'] = 'potential';
                            $json['message'] = 'Проверка прошла успешно!';
                            $json['type'] = 'success';

                            $this->updateConfigValue('telegram_key', $this->data['telegram_key']);
                            $this->updateConfigValue('webhook', 'potential');
                        }
                    }
                };
            }
        } else {
            if ((isset($response['url'])) && (($response['url']) == (GLOBAL_URL . '/api-telegram'))) {
                $json['webhook'] = 'tg://resolve?domain=' . $telegram->getBotInfo()['username'];

                $this->updateConfigValue('telegram_key', $this->data['telegram_key']);
                $this->updateConfigValue('webhook', $json['webhook']);
            } else {
                $json['webhook'] = 'potential';

                $this->updateConfigValue('telegram_key', $this->data['telegram_key']);
                $this->updateConfigValue('webhook', 'potential');
            }

            $json['message'] = 'Проверка прошла успешно!';
            $json['type'] = 'success';
        }

        if (!isset($json['type']) || $json['type'] != 'success') {
            $json['message'] = 'Неправильный Bot Token или недоступен мессенджер!';
            $json['type'] = 'error';
        }

        if (!isset($json['webhook'])) {
            $json['webhook'] = 'no';
        }



        return $json;
    }

    /**
     * @param string $key
     * @param $value
     * @return void
     */
    public function updateConfigValue(string $key, $value): void
    {
        // Проверка на пустой ключ
        if (empty($key)) {
            throw new \InvalidArgumentException('Ключ не может быть пустым.');
        }

        // Загружаем текущие данные
        $this->readTelegramArray();

        // Обновляем или добавляем значение
        $this->data[$key] = $value;

        // Сохраняем обновлённые данные
        $this->buildTelegramArray();
    }


    /**
     * Retrieves a list of template variables with their corresponding descriptions.
     *
     * @param array $additionalVariables An associative array of additional template variables to be merged with the default ones.
     * @return array An associative array of template variables where keys represent variable names and values represent their descriptions.
     */
    public function getTemplateVariables(array $additionalVariables = []): array
    {
        return array_merge([
            '{fullName}'     => 'Ф.И.О.',
            '{firstName}'    => 'Имя',
            '{lastName}'     => 'Фамилия',
            '{phone}'        => 'Телефон',
            '{email}'        => 'E-mail',
            '{message}'      => 'Сообщение',
            '{date_added}'   => 'Дата и время добавления',
        ], $additionalVariables);
    }


    /**
     * @param $filename
     * @return string
     */
    public function fetchLogData($filename): string
    {
        try {
            return $this->filesystem->get(GLOBAL_ROOT . '/temp/log/' . $filename);
        } catch (FileNotFoundException $e) {
            return '';
        }
    }

    /**
     * Очищает содержимое указанного файла.
     */
    public function clearFileContents(string $file): bool
    {
        // Попытка открытия файла
        $handle = @fopen($file, 'w+');
        if ($handle === false) {
            return false;
        }

        // Закрываем файл и возвращаем успешный результат
        fclose($handle);

        return true;
    }


}