<?php

namespace Mnv\Models;

use Mnv\Core\Model;
use Mnv\Core\Test\Logger;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NotFoundException;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\GroupDeletionException;
use Mnv\Core\Database\Throwable\DatabaseException;

/**
 * Class UserGroups
 * @package Mnv\Models\Users
 */
class UserGroups extends Model
{
    const ADMIN         = 'A';  // Администраторы
    const DEVELOPER     = 'D';  // Разработчики
    const MANAGER       = 'M';  // Менеджеры
    const MODERATOR     = 'O'; // Модератор
    const EDITOR        = 'E'; // Редактор
    const CONSULTANT    = 'K'; // Консультант
    const CUSTOMER      = 'C'; // Пользователи
    const SUBSCRIBER    = 'S'; // Подписчики


    /** @var string */
    protected string $table = 'user_group';
    protected string $table_privileges = 'user_group_privileges';

    /** @var string  */
    protected string $primaryKey = 'groupId';

    /** @var string */
    protected string $orderBy = 'groupId ASC';

    /** @var string */
    protected string $columns = 'groupId, hasAdminAccess, name, status, addedOn, modifiedOn';

    /** @var array|mixed  */
    public $privileges = array();

    /** @var array|string[]  */
    public array $groups = array(
        self::DEVELOPER     => 'DEVELOPER',  // Разработчики
        self::ADMIN         => 'ADMIN',      // Администраторы
        self::MANAGER       => 'MANAGER',    // Менеджеры
        self::MODERATOR     => 'MODERATOR',  // Модератор
        self::EDITOR        => 'EDITOR',     // Редактор
        self::CONSULTANT    => 'CONSULTANT', // Консультант
        self::CUSTOMER      => 'CUSTOMER',   // Пользователи
        self::SUBSCRIBER    => 'SUBSCRIBER', // Подписчики
    );

    public function __construct(Request $request)
    {
        $this->id           = $request->get('id');
        $this->data         = $request->get('group');
        $this->privileges   = $request->get('privileges');
    }

    /**
     * @param $limit
     * @param $page
     * @param $userType
     *
     * @return array|mixed
     */
    public function all($limit, $page, $userType = null)
    {

        if ($userType != 'D') connect()->notWhere($this->primaryKey, 1);

        return parent::all($limit, $page);
    }

    public function total($UserType = null): void
    {
        if ($UserType != 'D') connect()->notWhere($this->primaryKey, 1);

        parent::total();
    }

    /** Получить группу */
    public function edit(): UserGroups
    {
        if (empty($this->id)) {
            return $this;
        }

        if ($this->data = connect($this->table)->where($this->primaryKey, $this->id)->get('array')) {
            $this->data['countUsers'] = $this->getCountGroupUsers($this->id);
            $this->data['privilege'] = $this->getUserPrivileges();
        }

        return $this;
    }

    /**
     * Получить кол-во пользователей группы
     *
     * @param int $groupId
     *
     * @return int
     */
    public function getCountGroupUsers(int $groupId): int
    {
        return (int) connect('users')->count('*', 'count')->where('groupId', $groupId)->getValue() ?? 0;
    }

    private function getUserPrivileges(): ?array
    {
        return connect($this->table_privileges)->select('privilege')->where($this->primaryKey, $this->id)->pluck('privilege');
    }


    /**
     * Проверить существует группа или нет с таким `$fileName`
     *
     * @param string|null $fileName
     * @return mixed|string|null
     */
    public function checkFileName(?string $fileName): ?string
    {
        if (empty($fileName)) {
            return null;
        }

        if (!empty($this->id)) {
            connect()->where($this->primaryKey,'<>', $this->id);
        }

        $existingFileName = connect($this->table)->select('fileName')->where('LOWER(fileName)', strtolower($fileName))->getValue();

        return $existingFileName ?: null;
    }


    /**
     * @param array $data
     * @param int $managerId
     *
     * @return bool
     */
    public function prepare(array $data, int $managerId): bool
    {
        $currentTime = gmdate('Y-m-d H:i:s');

        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = $currentTime;

        if (empty($this->id)) {

            $data['addedBy']    = $managerId;
            $data['addedOn']    = $currentTime;
            $data['registered'] = $currentTime;

            if ($this->id = $this->insert($data)) {
                $this->privileges();
                return true;
            }
        } else {
            if ($this->update($data)) {
                $this->privileges();
                return true;
            }
        }

        return false;
    }

    /**
     * Сохранение доступа к пунктам меню (привелегии)
     * @return void
     */
    private function privileges(): void
    {
        if (!empty($this->privileges)) {
            connect($this->table_privileges)->where($this->primaryKey, $this->id)->delete();

            foreach ($this->privileges as $privilege) {
                connect($this->table_privileges)->insert([
                    'groupId' => $this->id,
                    'privilege' => $privilege
                ]);
            }
        }
    }

    /**
     * Delete a record
     *
     * @return bool
     *
     * @throws NoContentException
     * @throws NotFoundException
     * @throws DatabaseException
     * @throws GroupDeletionException
     */
    public function remove(): bool
    {
        // Проверка на недопустимые ID
        if (!empty($this->id) && in_array($this->id, [1, 2], true)) {
            throw new GroupDeletionException('Группа с ID ' . $this->id . ' не может быть удалена.');
        }

        if (parent::remove()) {
            // Удаление привилегий, связанных с группой
            connect($this->table_privileges)->where($this->primaryKey, $this->id)->delete();

            // Получение всех пользователей, связанных с данной группой
            $userIds = connect('users')
                ->where('groupId', $this->id)
                ->pluck('userId', 'userId');

            // Если найдены связанные пользователи, удаляем их записи
            if (!empty($userIds)) {
                connect('users')->in('userId', $userIds)->delete();
                connect('user_images')->in('userId', $userIds)->delete();
            }

            return true;
        }

        return false;
    }

    /**
     * Toggles the status of the current record between 'V' (Visible) and 'H' (Hidden).
     *
     * @return bool True if status was successfully toggled.
     *
     * @throws NotFoundException if `id` is missing or invalid.
     * @throws NoContentException if no record is found for the given `id`.
     * @throws GroupDeletionException
     */

    public function status(): bool
    {
        // Проверка на недопустимые ID
        if (!empty($this->id) && in_array($this->id, [1, 2], true)) {
            throw new GroupDeletionException('Статус группы с ID ' . $this->id . ' нельзя изменить.');
        }

        return parent::status();

    }



    /** STATIC  user_group */

    public static function selectGroup($groupId): array
    {
        $group = ['0' => "Выберите группу..."];
        if ($groupId != 1) {
            connect()->notWhere('groupId', 1);
        }

        $group += connect('user_group')
            ->select('groupId, name')
            ->where('status', 'V')
            ->pluck('name','groupId');

        return $group;
    }

    /**
     * Получаем все группы у кого `hasAdminAccess = 1` и статус  `status = V`
     *
     * @return array  Возвращаем `groupIds`
     */
    public static function arrayGroup(): array
    {
        return connect('user_group')
            ->select('groupId')
            ->where('hasAdminAccess', 1)
            ->where('status', 'V')
            ->pluck('groupId', 'groupId');
    }

    /**
     * @param $groupId
     * @return string|null
     */
    public static function groupName($groupId): ?string
    {
        return connect('user_group')->select('name')->where('groupId', $groupId)->getValue();
    }

    public static function getGroupLetter($groupId): ?string
    {
        return connect('user_group')->select('fileName')->where('groupId', $groupId)->getValue();
    }
}