<?php
namespace Mnv\Modules;

use Mnv\Core\Locale\I18N;
use Mnv\Models\UserGroups;

require_once GLOBAL_ROOT . '/includes/admin-functions.inc.php';

/**
 * Class Newsletter
 * @package Mnv\Modules
 */
class Newsletter
{
    /** @var string Таблица с подписками */
    protected string $table = 'subscribe';

    /** @var array|null Ответ с результатом выполнения */
    public ?array $response = null;

    /**
     * Регистрирует новый email для подписки.
     *
     * @param string $email Адрес электронной почты для подписки.
     * @return void
     */
    public function register(string $email): void
    {
        // Валидация email
        if (!$this->isValidEmail($email)) {
            return;
        }

        // Проверка наличия email в базе данных
        if ($this->emailExists($email)) {
            $this->response = array('status' => 403, 'type' => 'error', 'message' => I18N::locale(
                "Это email уже подписан",
                "Bu elektron pochta allaqachon imzolangan",
                "This email is already signed"
            ));
            return;
        }

        // Вставка нового email
        if ($this->insertEmail($email)) {
            $this->response = array('status' => 200, 'type' => 'success', 'message' => I18N::locale(
                "Вы успешно подписаны",
                "Siz muvaffaqiyatli obuna bo'ldingiz",
                "You have been successfully subscribed"
            ));
        } else {
            $this->response = array('status' => 403, 'type' => 'error', 'message' => I18N::locale(
                "Произошла ошибка. Пожалуйста, попробуйте еще раз",
                "Xato yuz berdi. Iltimos, yana urinib ko'ring",
                "An error has occurred. Please try again"
            ));
        }
    }

    /**
     * Проверяет корректность email.
     *
     * @param string $email
     * @return bool
     */
    private function isValidEmail(string $email): bool
    {
        if (empty($email)) {
            $this->response = array('status' => 403, 'type' => 'error', 'message' => I18N::locale(
                "Поле email адреса не должно быть пустым.",
                "Elektron pochta manzili maydoni bo'sh bo'lmasligi kerak.",
                "The email address field should not be empty."
            ));
            return false;
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->response = array('status' => 403, 'type' => 'error', 'message' => I18N::locale(
                "Вы должны заполнить поле действительным адресом электронной почты",
                "Siz maydonni haqiqiy elektron pochta manzili bilan to'ldirishingiz kerak",
                "You must fill in the field with a valid email address"
            ));
            return false;
        }

        return true;
    }

    /**
     * Проверяет, существует ли email в базе данных.
     *
     * @param string $email
     * @return bool
     */
    private function emailExists(string $email): bool
    {
        try {
            $userExists = connect($this->table)->count('*')->where('email', $email)->get();

            return $userExists > 0;
        } catch (\Exception $e) {
            $this->response = array('status' => 500, 'type' => 'error', 'message' => I18N::locale(
                'Ошибка при проверке email',
                "Elektron pochtani tekshirishda xato",
                "Error while checking email"
            ));
            return false;
        }
    }

    /**
     * Вставляет новый email в базу данных.
     *
     * @param string $email
     * @return bool
     */
    private function insertEmail(string $email): bool
    {
        try {
            return connect($this->table)->insert([
                'email' => $email,
                'subscribedAt' => gmdate('Y-m-d H:i:s')
            ]);
        } catch (\Exception $e) {
            error_log('Ошибка при добавлении email: ' . $e->getMessage()); // Логируем ошибку
            return false;
        }
    }


}