<?php

namespace Mnv\Modules;

class Shops
{
    private string $orderBy = 'publishedOn ASC';

    /** @var array FeatureCollection данных магазинов */
    public array $shops = [];
    public array $cities = [];

    /** @var array Список ID секций */
    private array $sectionIds = [];

    /**
     * Конструктор.
     *
     * @param int|string|null $sectionId ID секции для обработки

     */
    public function __construct($sectionId)
    {
        global $SECTIONS;

        if (empty($sectionId) || empty($SECTIONS[$sectionId])) {
            return;
        }

        // Формируем список ID секций (включая под-дети)
        $this->sectionIds = [$sectionId];
        if (!empty($SECTIONS[$sectionId]['allChildren'])) {
            $this->sectionIds = array_merge($this->sectionIds, $SECTIONS[$sectionId]['allChildren']);
        }

        // Настройка подключения с фильтрацией по секциям
        connect()->in('sectionId', $this->sectionIds);
    }

    /**
     * Загружает все магазины и формирует FeatureCollection.
     *
     * @return void
     */
    public function all(): void
    {
        if (empty($this->sectionIds)) {
            return;
        }

        $cities = connect('sections')
            ->select('sectionId AS city_id, name, alias')
            ->where('parentId', 6)
            ->orderBy('sortOrder ASC')
            ->getAll('array');

        $this->cities = collect($cities)->map(function ($city) {
            return [
                'city_id' => $city['city_id'],
                'city_name' => $city['name'],
                'cord_center' => !empty($city['alias'])
                    ? explode(', ', $city['alias'])
                    : [0, 0],
            ];
        })->toArray();


        // Инициализация структуры GeoJSON
        $this->shops = [
            'type' => "FeatureCollection",
            'features' => []
        ];

        // Получение списка магазинов
        $shops = connect('articles')
            ->select('articleId, title, alias')
            ->in('sectionId', $this->sectionIds)
            ->where('contentType', 'shops')
            ->orderBy($this->orderBy)
            ->limit(1000)
            ->getAll('array');

        foreach ($shops as $shop) {
            $this->addShopFeature($shop);
        }
    }

    /**
     * Получает свойства для статьи и добавляет их в FeatureCollection.
     *
     * @param array $shop Информация о магазине
     * @return void
     */
    private function addShopFeature(array $shop): void
    {
        $properties = $this->getProperties($shop['articleId']);
        if (empty($properties['shop_coordinates'])) {
            return;
        }

        $coordinates = explode(",", $properties['shop_coordinates']);

        // Проверка корректности координат
        if (count($coordinates) !== 2 || !is_numeric($coordinates[0]) || !is_numeric($coordinates[1])) {
            return;
        }

        // Добавляем магазин в FeatureCollection
        $this->shops['features'][] = [
            'id' => $shop['articleId'],
            'type' => "Feature",
            'geometry' => [
                'type' => "Point",
                'coordinates' => array_map('floatval', $coordinates)
            ],
            'properties' => [
                'balloonContent' => $shop['title'],
                'id' => $shop['articleId']
            ]
        ];
    }

    /**
     * Получает пользовательские свойства для статьи.
     *
     * @param int $articleId ID статьи
     * @return array Ассоциативный массив свойств
     */
    private function getProperties(int $articleId): array
    {
        return connect('type_content_custom_values')
            ->usingJoin('type_content_custom_fields', 'fileName')
            ->select('articleId, typeId, type, fieldId, fileName, name, value, orderBy')
            ->where('articleId', $articleId)
            ->where('status', 'V')
            ->orderBy('orderBy')
            ->pluck('value', 'fileName') ?? [];
    }
}